package ru.yandex.direct.intapi.entity.metrika.model.objectinfo;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;

import static java.util.Comparator.comparing;

/**
 * Ответ ручки object_info для всех типов объектов.
 * <p>
 * Содержит список объектов и токен для получения следующей порции объектов.
 * Если по полученному токену для извлечения данной порции объектов извлечен
 * хотябы один объект, то для ответа вычисляется новый токен, по которому
 * будет запрашиваться следующая порция. Если же ни одного объекта не извлечено,
 * то в ответе отдается полученные токен как есть.
 *
 * @param <T> тип возвращаемых объектов.
 */
@ParametersAreNonnullByDefault
public class MetrikaObjectInfoResponse<T extends MetrikaObjectInfo> {

    static final Comparator<MetrikaObjectInfo> COMPARATOR = comparing(MetrikaObjectInfo::getSortLastChange)
            .thenComparing(MetrikaObjectInfo::getSortId);

    /**
     * Список возвращаемых объектов.
     */
    @JsonProperty("objects")
    private List<T> objects;

    /**
     * Тайм-токен, который необходимо запомнить клиенту для запроса следующей порции объектов.
     */
    @JsonProperty("time_token")
    private String timeToken;

    @JsonCreator
    private MetrikaObjectInfoResponse(@JsonProperty("objects") List<T> objects,
                                      @JsonProperty("time_token") @Nullable String timeToken) {
        this.objects = objects;
        this.timeToken = timeToken;
    }

    /**
     * Создает ответ ручки object_info из списка извлеченных из БД объектов,
     * лимита и тайм-токена, использованного для получения этой порции объектов.
     * <p>
     * В некоторых случаях лимит может быть меньше, чем извлеченный список объектов,
     * и применяется после сортировки. Это происходит, когда извлекаются объекты, являющиеся
     * различными на уровне базы, и извлекаются из разных таблиц. Тогда оба типа объектов
     * извлекаются с указанным лимитом, и объектов извлекается условно в два раза больше.
     * И только после проведения сквозной сортировки общего списка объектов можно обрезать лишние.
     * <p>
     * Тайм-токен, использованный для получения этой порции объектов, необходим для ситуации,
     * когда ни одного объекта по нему не извлечено, тогда он должен быть отдан в ответе
     * для получения следующей порции.
     *
     * @param objects       список извлеченных из базы объектов.
     * @param limit         ограничение на количество объектов в ответе, применяется после сортировки.
     * @param prevTimeToken значение тайм-токена, использованное для получения этого списка объектов.
     * @param <T>           тип объектов.
     * @return ответ ручки object_info.
     */
    public static <T extends MetrikaObjectInfo> MetrikaObjectInfoResponse<T> create(
            List<T> objects, @Nullable Integer limit, @Nullable MetrikaTimeToken prevTimeToken) {
        objects = new ArrayList<>(objects);
        objects.sort(COMPARATOR);
        if (limit != null && limit < objects.size()) {
            objects = objects.subList(0, limit);
        }

        MetrikaTimeToken nextTimeToken = calcNextTimeToken(prevTimeToken, objects);
        String nextTimeTokenStr = nextTimeToken != null ? nextTimeToken.toString() : null;
        return new MetrikaObjectInfoResponse<>(objects, nextTimeTokenStr);
    }

    public List<T> getObjects() {
        return objects;
    }

    public String getTimeToken() {
        return timeToken;
    }

    @Nullable
    private static MetrikaTimeToken calcNextTimeToken(@Nullable MetrikaTimeToken prevTimeToken,
                                                      List<? extends MetrikaObjectInfo> sortedObjectsInfo) {
        if (sortedObjectsInfo.isEmpty()) {
            return prevTimeToken;
        }
        MetrikaObjectInfo lastObjectInfo = sortedObjectsInfo.get(sortedObjectsInfo.size() - 1);
        return new MetrikaTimeToken(lastObjectInfo.getSortLastChange(), lastObjectInfo.getSortId());
    }
}
