package ru.yandex.direct.intapi.entity.metrika.model.objectinfo;

import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;

import javax.annotation.ParametersAreNonnullByDefault;

/**
 * Временной токен. Используется для инкрементального экспорта объектов,
 * изменившихся с определенного момента времени, на который указывает токен.
 * <p>
 * Первая порция объектов запрашивается без токена, в ответе отправляются
 * объекты, отсортированные по (времени изменения + id) по возрастанию.
 * Вместе с объектами отправляется токен для запроса следующей порции объектов,
 * состоящий из времени изменения и id последнего объекта после сортировки
 * по этим параметрам.
 * <p>
 * Следующая порция запрашивается с этим токеном и выбираются объекты,
 * изменившиеся после указанного момента времени, или в этот же момент,
 * но имеющие большее значение id.
 * <p>
 * Например:
 * <p>
 * 1 итерация:
 * запрос: token = null, limit = 3
 * ответ:
 * {
 * objects: [
 * {
 * LastChange: 2017-01-10T15:00:00
 * id: 30
 * },
 * {
 * LastChange: 2017-01-15T19:20:00
 * id: 15
 * },
 * {
 * LastChange: 2017-01-15T19:20:00
 * id: 18
 * }
 * ],
 * timeToken: "2017-01-15T19:20:00/18"
 * }
 * <p>
 * 2 итерация:
 * запрос: token = "2017-01-15T19:20:00/18", limit = 3
 * ответ:
 * {
 * objects: [
 * {
 * LastChange: 2017-01-15T19:20:00
 * id: 20
 * },
 * {
 * LastChange: 2017-01-20T14:40:00
 * id: 17
 * }
 * ],
 * timeToken: "2017-01-15T19:20:00/18"
 * }
 */
@ParametersAreNonnullByDefault
public class MetrikaTimeToken {

    public static final String SEPARATOR = "/";

    // 2013-02-13T12:35:07
    public static final DateTimeFormatter DATE_TIME_FORMATTER = DateTimeFormatter.ISO_LOCAL_DATE_TIME;

    private LocalDateTime lastChange;
    private Long lastId;

    public MetrikaTimeToken(LocalDateTime lastChange, Long lastId) {
        this.lastChange = lastChange;
        this.lastId = lastId;
    }

    public MetrikaTimeToken(String timeToken) {
        this(extractDateTimeFromTimeToken(timeToken), extractLastIdFromTimeToken(timeToken));
    }

    public LocalDateTime getLastChange() {
        return lastChange;
    }

    public Long getLastId() {
        return lastId;
    }

    public String toString() {
        return lastChange.format(DATE_TIME_FORMATTER) + SEPARATOR + lastId;
    }

    public static String extractRawDateTimeFromTimeToken(String timeToken) {
        int sepIndex = timeToken.indexOf(SEPARATOR);
        return timeToken.substring(0, sepIndex);
    }

    public static String extractRawLastIdFromTimeToken(String timeToken) {
        int sepIndex = timeToken.indexOf(SEPARATOR);
        return timeToken.substring(sepIndex + 1);
    }

    private static LocalDateTime extractDateTimeFromTimeToken(String timeToken) {
        return LocalDateTime.parse(extractRawDateTimeFromTimeToken(timeToken), DATE_TIME_FORMATTER);
    }

    private static Long extractLastIdFromTimeToken(String timeToken) {
        return Long.valueOf(extractRawLastIdFromTimeToken(timeToken));
    }
}
