package ru.yandex.direct.intapi.entity.metrika.service;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.Function;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.metrika.model.BannerForMetrika;
import ru.yandex.direct.core.entity.metrika.repository.MetrikaBannerRepository;
import ru.yandex.direct.dbutil.SqlUtils;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.intapi.entity.metrika.model.MetrikaBannersParam;
import ru.yandex.direct.intapi.entity.metrika.model.MetrikaBannersResult;
import ru.yandex.direct.intapi.validation.IntApiDefect;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.intapi.validation.IntApiConstraints.notEmptyCollection;
import static ru.yandex.direct.intapi.validation.IntApiConstraints.notNull;
import static ru.yandex.direct.intapi.validation.IntApiConstraints.validId;
import static ru.yandex.direct.intapi.validation.ValidationUtils.checkResult;

@Service
public class MetrikaBannersService {

    @Autowired
    private MetrikaBannerRepository metrikaBannerRepository;

    @Autowired
    private CampaignRepository campaignRepository;

    @Autowired
    private ShardHelper shardHelper;

    public List<MetrikaBannersResult> getBanners(List<MetrikaBannersParam> params) {
        ValidationResult<List<MetrikaBannersParam>, IntApiDefect> validationResult = validate(params);
        checkResult(validationResult);

        return shardHelper.groupByShard(params, ShardKey.ORDER_ID, MetrikaBannersParam::getOrderId)
                .chunkedBy(SqlUtils.TYPICAL_SELECT_CHUNK_SIZE)
                .stream()
                .map(e -> toResult(e.getKey(), e.getValue()))
                .toFlatCollection(Function.identity(), ArrayList::new);
    }

    private List<MetrikaBannersResult> toResult(Integer shard, List<MetrikaBannersParam> params) {
        Map<Long, Long> oid2cid = campaignRepository.getCidsForOrderIds(shard, toOrderIdsList(params));
        List<MetrikaBannersResult> results = new ArrayList<>();

        List<Long> bannerIds = toBannerIdsList(params);

        Map<Long, BannerForMetrika> banners = metrikaBannerRepository.getBanners(shard, bannerIds);

        bannerIds = StreamEx.of(bannerIds)
                .filter(id -> !banners.containsKey(id))
                .toList();

        if (bannerIds.size() > 0) {
            banners.putAll(metrikaBannerRepository.getImageBanners(shard, bannerIds));
        }

        for (MetrikaBannersParam param : params) {
            BannerForMetrika banner = banners.get(param.getBannerId());
            if (banner != null && Objects.equals(oid2cid.get(param.getOrderId()), banner.getCid())) {
                results.add(convertToResult(banner, param));
            }
        }
        return results;
    }

    private MetrikaBannersResult convertToResult(BannerForMetrika banner, MetrikaBannersParam param) {
        return (MetrikaBannersResult) new MetrikaBannersResult()
                .withBid(banner.getBid())
                .withBody(banner.getBody())
                .withTitle(banner.getTitle())
                .withDomain(banner.getDomain())
                .withIsImageBanner(banner.getIsImage())
                .withBannerId(param.getBannerId())
                .withOrderId(param.getOrderId());
    }

    private List<Long> toBannerIdsList(List<MetrikaBannersParam> params) {
        return StreamEx.of(params).map(MetrikaBannersParam::getBannerId).toList();
    }

    private List<Long> toOrderIdsList(List<MetrikaBannersParam> params) {
        return StreamEx.of(params).map(MetrikaBannersParam::getOrderId).toList();
    }

    protected ValidationResult<List<MetrikaBannersParam>, IntApiDefect> validate(
            List<MetrikaBannersParam> params) {
        ListValidationBuilder<MetrikaBannersParam, IntApiDefect> v =
                ListValidationBuilder.of(params, IntApiDefect.class);

        v.check(notNull());
        v.check(notEmptyCollection());
        v.checkEach(notNull());
        v.checkEachBy(this::keyIsValid);

        return v.getResult();
    }

    private ValidationResult<MetrikaBannersParam, IntApiDefect> keyIsValid(
            MetrikaBannersParam key) {

        ItemValidationBuilder<MetrikaBannersParam, IntApiDefect> v =
                ItemValidationBuilder.of(key, IntApiDefect.class);

        if (key == null) {
            return v.getResult();
        }

        v.item(key.getOrderId(), "order_id")
                .check(notNull())
                .check(validId());
        v.item(key.getBannerId(), "banner_id")
                .check(notNull())
                .check(validId());

        return v.getResult();
    }

}
