package ru.yandex.direct.intapi.entity.metrika.service;

import java.util.Collection;
import java.util.List;
import java.util.function.Function;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.metrika.model.CampaignForMetrika;
import ru.yandex.direct.core.entity.metrika.repository.MetrikaCampaignRepository;
import ru.yandex.direct.dbutil.SqlUtils;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.intapi.entity.metrika.model.MetrikaCampaignsResult;
import ru.yandex.direct.intapi.validation.IntApiDefect;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.intapi.validation.IntApiConstraints.notEmptyCollection;
import static ru.yandex.direct.intapi.validation.IntApiConstraints.notNull;
import static ru.yandex.direct.intapi.validation.IntApiConstraints.validId;
import static ru.yandex.direct.intapi.validation.ValidationUtils.checkResult;

@Service
public class MetrikaCampaignsService {

    @Autowired
    private ShardHelper shardHelper;
    @Autowired
    private MetrikaCampaignRepository metrikaCampaignRepository;

    /**
     * Получение информации о кампаниях по order_id
     */
    public List<MetrikaCampaignsResult> getCampaigns(List<Long> orderIds) {
        ValidationResult<List<Long>, IntApiDefect> validationResult = validate(orderIds);
        checkResult(validationResult);

        return shardHelper.groupByShard(orderIds, ShardKey.ORDER_ID, Function.identity())
                .chunkedBy(SqlUtils.TYPICAL_SELECT_CHUNK_SIZE)
                .stream()
                .mapKeyValue(metrikaCampaignRepository::getCampaignsForMetrika)
                .flatMap(Collection::stream)
                .map(this::toMetrikaResult)
                .toList();
    }

    public ValidationResult<List<Long>, IntApiDefect> validate(List<Long> orderIds) {
        ListValidationBuilder<Long, IntApiDefect> v =
                ListValidationBuilder.of(orderIds, IntApiDefect.class);

        v.check(notNull());
        v.check(notEmptyCollection());
        v.checkEach(notNull());
        v.checkEachBy(this::orderIdIsValid);

        return v.getResult();
    }

    private MetrikaCampaignsResult toMetrikaResult(CampaignForMetrika camp) {
        return new MetrikaCampaignsResult()
                .withCid(camp.getId())
                .withClientId(camp.getClientId())
                .withCurrency(camp.getCurrency().toString())
                .withMetrikaCounters(camp.getMetrikaCounters())
                .withName(camp.getName())
                .withOrderId(camp.getOrderId());
    }

    private ValidationResult<Long, IntApiDefect> orderIdIsValid(Long orderId) {

        ItemValidationBuilder<Long, IntApiDefect> v =
                ItemValidationBuilder.of(orderId, IntApiDefect.class);

        if (orderId == null) {
            return v.getResult();
        }

        v.check(notNull())
                .check(validId());

        return v.getResult();
    }
}
