package ru.yandex.direct.intapi.entity.metrika.service;

import java.util.ArrayList;
import java.util.List;
import java.util.function.Function;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.IdModFilter;
import ru.yandex.direct.core.entity.metrika.model.CampaignWithCounterForMetrika;
import ru.yandex.direct.core.entity.metrika.repository.MetrikaCampaignRepository;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.intapi.entity.metrika.model.MetrikaCountersResult;
import ru.yandex.direct.intapi.validation.IntApiDefect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.intapi.validation.ValidationUtils.checkResult;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
public class MetrikaCountersService {

    @Autowired
    private ShardHelper shardHelper;

    @Autowired
    private MetrikaCampaignRepository metrikaCampaignRepository;

    @Autowired
    private ModFilterValidationService modFilterValidationService;

    public List<MetrikaCountersResult> getCounters(long divisor, long reminder) {
        IdModFilter campaignIdfilter = new IdModFilter(divisor, reminder);

        ValidationResult<IdModFilter, IntApiDefect> validationResult =
                modFilterValidationService.validate(campaignIdfilter);
        checkResult(validationResult);

        return getMetrikaCounters(campaignIdfilter);
    }


    private List<MetrikaCountersResult> getMetrikaCounters(IdModFilter filter) {
        List<MetrikaCountersResult> countersResults = new ArrayList<>();
        shardHelper.forEachShard((shard) -> {
            countersResults.addAll(
                    StreamEx.of(metrikaCampaignRepository.getCampaignsWithCounters(shard, filter))
                            .map(this::toMetrikaResult)
                            .toFlatCollection(Function.identity(), ArrayList::new)
            );
        });
        return countersResults;
    }

    private List<MetrikaCountersResult> toMetrikaResult(CampaignWithCounterForMetrika camp) {
        return mapList(camp.getMetrikaCounters(), c -> convertToMetrikaCountersResult(camp, c));
    }

    private MetrikaCountersResult convertToMetrikaCountersResult(CampaignWithCounterForMetrika camp, Long counterId) {
        return new MetrikaCountersResult()
                .withCid(camp.getId())
                .withOrderId(camp.getOrderId())
                .withCounterId(counterId);
    }
}
