package ru.yandex.direct.intapi.entity.metrika.service;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.function.Function;

import one.util.streamex.StreamEx;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.metrika.model.DynamicForMetrika;
import ru.yandex.direct.core.entity.metrika.repository.MetrikaDynamicRepository;
import ru.yandex.direct.dbutil.SqlUtils;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.intapi.entity.metrika.model.MetrikaDynamicsParam;
import ru.yandex.direct.intapi.entity.metrika.model.MetrikaDynamicsResult;
import ru.yandex.direct.intapi.validation.IntApiDefect;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.intapi.validation.IntApiConstraints.notEmptyCollection;
import static ru.yandex.direct.intapi.validation.IntApiConstraints.notNull;
import static ru.yandex.direct.intapi.validation.IntApiConstraints.validId;
import static ru.yandex.direct.intapi.validation.ValidationUtils.checkResult;

@Service
public class MetrikaDynamicsService {

    @Autowired
    private ShardHelper shardHelper;
    @Autowired
    private MetrikaDynamicRepository metrikaDynamicRepository;

    public List<MetrikaDynamicsResult> getDynamics(List<MetrikaDynamicsParam> params) {
        ValidationResult<List<MetrikaDynamicsParam>, IntApiDefect> validationResult = validate(params);
        checkResult(validationResult);

        return shardHelper.groupByShard(params, ShardKey.ORDER_ID, MetrikaDynamicsParam::getOrderId)
                .chunkedBy(SqlUtils.TYPICAL_SELECT_CHUNK_SIZE)
                .stream()
                .map(e -> toResult(e.getKey(), e.getValue()))
                .toFlatCollection(Function.identity(), ArrayList::new);
    }

    private List<MetrikaDynamicsResult> toResult(Integer shard,
                                                 List<MetrikaDynamicsParam> params) {
        List<Pair<Long, Long>> pairs = StreamEx.of(params)
                .map(p -> Pair.of(p.getOrderId(), p.getDynCondId()))
                .toList();

        Map<Long, DynamicForMetrika>
                existingFilters = metrikaDynamicRepository.getConditions(shard, pairs);

        List<MetrikaDynamicsResult> result = new ArrayList<>();
        for (MetrikaDynamicsParam param : params) {
            DynamicForMetrika dynamicForMetrika = existingFilters.get(param.getDynCondId());
            if (dynamicForMetrika != null) {
                result.add(convertToResult(dynamicForMetrika, param));
            }
        }
        return result;
    }

    private MetrikaDynamicsResult convertToResult(DynamicForMetrika dynamicForMetrika, MetrikaDynamicsParam param) {
        return (MetrikaDynamicsResult) new MetrikaDynamicsResult()
                .withConditionName(dynamicForMetrika.getConditionName())
                .withDynCondId(param.getDynCondId())
                .withOrderId(param.getOrderId());
    }

    protected ValidationResult<List<MetrikaDynamicsParam>, IntApiDefect> validate(
            List<MetrikaDynamicsParam> params) {
        ListValidationBuilder<MetrikaDynamicsParam, IntApiDefect> v =
                ListValidationBuilder.of(params, IntApiDefect.class);

        v.check(notNull());
        v.check(notEmptyCollection());
        v.checkEach(notNull());
        v.checkEachBy(this::keyIsValid);

        return v.getResult();
    }

    private ValidationResult<MetrikaDynamicsParam, IntApiDefect> keyIsValid(
            MetrikaDynamicsParam key) {

        ItemValidationBuilder<MetrikaDynamicsParam, IntApiDefect> v =
                ItemValidationBuilder.of(key, IntApiDefect.class);

        if (key == null) {
            return v.getResult();
        }

        v.item(key.getOrderId(), "order_id")
                .check(notNull())
                .check(validId());
        v.item(key.getDynCondId(), "dyn_cond_id")
                .check(notNull())
                .check(validId());

        return v.getResult();
    }

}
