package ru.yandex.direct.intapi.entity.metrika.service;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.Function;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.metrika.model.GroupForMetrika;
import ru.yandex.direct.core.entity.metrika.repository.MetrikaAdGroupRepository;
import ru.yandex.direct.dbutil.SqlUtils;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.intapi.entity.metrika.model.MetrikaGroupsParam;
import ru.yandex.direct.intapi.entity.metrika.model.MetrikaGroupsResult;
import ru.yandex.direct.intapi.validation.IntApiDefect;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.intapi.validation.IntApiConstraints.notEmptyCollection;
import static ru.yandex.direct.intapi.validation.IntApiConstraints.notNull;
import static ru.yandex.direct.intapi.validation.IntApiConstraints.validId;
import static ru.yandex.direct.intapi.validation.ValidationUtils.checkResult;

@Service
public class MetrikaGroupsService {

    @Autowired
    private ShardHelper shardHelper;

    @Autowired
    private MetrikaAdGroupRepository metrikaAdGroupRepository;

    @Autowired
    private CampaignRepository campaignRepository;

    public List<MetrikaGroupsResult> getGroups(List<MetrikaGroupsParam> params) {
        ValidationResult<List<MetrikaGroupsParam>, IntApiDefect> validationResult = validate(params);
        checkResult(validationResult);

        return shardHelper.groupByShard(params, ShardKey.ORDER_ID, MetrikaGroupsParam::getOrderId)
                .chunkedBy(SqlUtils.TYPICAL_SELECT_CHUNK_SIZE)
                .stream()
                .map(e -> toResult(e.getKey(), e.getValue()))
                .toFlatCollection(Function.identity(), ArrayList::new);
    }

    private List<MetrikaGroupsResult> toResult(Integer shard, List<MetrikaGroupsParam> params) {
        List<MetrikaGroupsResult> results = new ArrayList<>();
        Map<Long, Long> oid2cid = campaignRepository.getCidsForOrderIds(shard, toOrderIdsList(params));
        Map<Long, GroupForMetrika> existingGroups = metrikaAdGroupRepository.getGroups(shard, toGroupIdsList(params));
        for (MetrikaGroupsParam param : params) {
            GroupForMetrika group = existingGroups.get(param.getGroupId());
            if (group != null && Objects.equals(group.getCid(), oid2cid.get(param.getOrderId()))) {
                results.add(convertToResult(group, param));
            }
        }
        return results;
    }

    private MetrikaGroupsResult convertToResult(GroupForMetrika groupForMetrika, MetrikaGroupsParam param) {
        return (MetrikaGroupsResult) new MetrikaGroupsResult()
                .withGroupName(groupForMetrika.getGroupName())
                .withGroupId(param.getGroupId())
                .withOrderId(param.getOrderId());
    }

    private List<Long> toOrderIdsList(List<MetrikaGroupsParam> params) {
        return StreamEx.of(params).map(MetrikaGroupsParam::getOrderId).toList();
    }

    private List<Long> toGroupIdsList(List<MetrikaGroupsParam> params) {
        return StreamEx.of(params).map(MetrikaGroupsParam::getGroupId).toList();
    }

    protected ValidationResult<List<MetrikaGroupsParam>, IntApiDefect> validate(List<MetrikaGroupsParam> params) {
        ListValidationBuilder<MetrikaGroupsParam, IntApiDefect> v =
                ListValidationBuilder.of(params, IntApiDefect.class);

        v.check(notNull());
        v.check(notEmptyCollection());
        v.checkEach(notNull());
        v.checkEachBy(this::keyIsValid);

        return v.getResult();
    }

    private ValidationResult<MetrikaGroupsParam, IntApiDefect> keyIsValid(
            MetrikaGroupsParam key) {

        ItemValidationBuilder<MetrikaGroupsParam, IntApiDefect> v =
                ItemValidationBuilder.of(key, IntApiDefect.class);

        if (key == null) {
            return v.getResult();
        }

        v.item(key.getOrderId(), "order_id")
                .check(notNull())
                .check(validId());
        v.item(key.getGroupId(), "group_id")
                .check(notNull())
                .check(validId());

        return v.getResult();
    }

}
