package ru.yandex.direct.intapi.entity.metrika.service;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.function.Function;

import one.util.streamex.StreamEx;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.metrika.model.PerformanceFilterForMetrika;
import ru.yandex.direct.core.entity.metrika.repository.MetrikaPerformanceFilterRepository;
import ru.yandex.direct.dbutil.SqlUtils;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.intapi.entity.metrika.model.MetrikaPerformanceFiltersParam;
import ru.yandex.direct.intapi.entity.metrika.model.MetrikaPerformanceFiltersResult;
import ru.yandex.direct.intapi.validation.IntApiDefect;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.intapi.validation.IntApiConstraints.notEmptyCollection;
import static ru.yandex.direct.intapi.validation.IntApiConstraints.notNull;
import static ru.yandex.direct.intapi.validation.IntApiConstraints.validId;
import static ru.yandex.direct.intapi.validation.ValidationUtils.checkResult;

@Service
public class MetrikaPerformanceFiltersService {

    @Autowired
    private ShardHelper shardHelper;

    @Autowired
    private MetrikaPerformanceFilterRepository metrikaPerformanceFilterRepository;

    public List<MetrikaPerformanceFiltersResult> getPerformanceFilters(
            List<MetrikaPerformanceFiltersParam> params) {
        ValidationResult<List<MetrikaPerformanceFiltersParam>, IntApiDefect> validationResult = validate(params);
        checkResult(validationResult);

        return shardHelper.groupByShard(params, ShardKey.ORDER_ID, MetrikaPerformanceFiltersParam::getOrderId)
                .chunkedBy(SqlUtils.TYPICAL_SELECT_CHUNK_SIZE)
                .stream()
                .map(e -> toResult(e.getKey(), e.getValue()))
                .toFlatCollection(Function.identity(), ArrayList::new);
    }

    private List<MetrikaPerformanceFiltersResult> toResult(Integer shard,
                                                           List<MetrikaPerformanceFiltersParam> params) {
        List<Pair<Long, Long>> pairs = StreamEx.of(params)
                .map(p -> Pair.of(p.getOrderId(), p.getPerfFilterId()))
                .toList();

        Map<Long, PerformanceFilterForMetrika>
                existingFilters = metrikaPerformanceFilterRepository.getFilters(shard, pairs);

        List<MetrikaPerformanceFiltersResult> result = new ArrayList<>();
        for (MetrikaPerformanceFiltersParam param : params) {
            PerformanceFilterForMetrika filterForMetrika = existingFilters.get(param.getPerfFilterId());
            if (filterForMetrika != null) {
                result.add(convertToResult(filterForMetrika, param));
            }
        }
        return result;
    }

    private MetrikaPerformanceFiltersResult convertToResult(PerformanceFilterForMetrika filterForMetrika,
                                                            MetrikaPerformanceFiltersParam param) {
        return (MetrikaPerformanceFiltersResult) new MetrikaPerformanceFiltersResult()
                .withName(filterForMetrika.getFilterName())
                .withPerfFilterId(param.getPerfFilterId())
                .withOrderId(param.getOrderId());
    }

    protected ValidationResult<List<MetrikaPerformanceFiltersParam>, IntApiDefect> validate(
            List<MetrikaPerformanceFiltersParam> params) {
        ListValidationBuilder<MetrikaPerformanceFiltersParam, IntApiDefect> v =
                ListValidationBuilder.of(params, IntApiDefect.class);

        v.check(notNull());
        v.check(notEmptyCollection());
        v.checkEach(notNull());
        v.checkEachBy(this::keyIsValid);

        return v.getResult();
    }

    private ValidationResult<MetrikaPerformanceFiltersParam, IntApiDefect> keyIsValid(
            MetrikaPerformanceFiltersParam key) {

        ItemValidationBuilder<MetrikaPerformanceFiltersParam, IntApiDefect> v =
                ItemValidationBuilder.of(key, IntApiDefect.class);

        if (key == null) {
            return v.getResult();
        }

        v.item(key.getOrderId(), "order_id")
                .check(notNull())
                .check(validId());
        v.item(key.getPerfFilterId(), "perf_filter_id")
                .check(notNull())
                .check(validId());

        return v.getResult();
    }

}
