package ru.yandex.direct.intapi.entity.metrika.service;

import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.IdModFilter;
import ru.yandex.direct.intapi.validation.IntApiDefect;
import ru.yandex.direct.intapi.validation.IntApiDefects;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.validation.Predicates.greaterThan;
import static ru.yandex.direct.validation.Predicates.greaterThanOrEqualTo;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;

@Service
public class ModFilterValidationService {

    private static final Long DIVISOR_MUST_BE_GREATER_THAN = 1L;
    private static final Long REMINDER_MUST_BE_GREATER_THAN_OR_EQUAL_TO = 0L;

    private static IntApiDefect divisorIsGreaterThanReminderDefect() {
        return (defectInfo) -> "divisor must be greater than reminder";
    }

    public ValidationResult<IdModFilter, IntApiDefect> validate(IdModFilter filter) {
        ItemValidationBuilder<IdModFilter, IntApiDefect> v =
                ItemValidationBuilder.of(filter, IntApiDefect.class);

        v.item(filter.getDivisor(), "divisor")
                .check(fromPredicate(greaterThan(DIVISOR_MUST_BE_GREATER_THAN), IntApiDefects.greaterThan(
                        DIVISOR_MUST_BE_GREATER_THAN)));

        v.item(filter.getRemainder(), "reminder")
                .check(fromPredicate(greaterThanOrEqualTo(REMINDER_MUST_BE_GREATER_THAN_OR_EQUAL_TO),
                        IntApiDefects.greaterThanOrEqualTo(REMINDER_MUST_BE_GREATER_THAN_OR_EQUAL_TO)));

        v.check(divisorIsGreaterThanReminder(), When.isValid());

        return v.getResult();
    }

    private Constraint<IdModFilter, IntApiDefect> divisorIsGreaterThanReminder() {
        return fromPredicate(goalsFilter -> goalsFilter.getDivisor() > goalsFilter.getRemainder(),
                divisorIsGreaterThanReminderDefect());
    }
}
