package ru.yandex.direct.intapi.entity.metrika.service.objectinfo.validation;

import java.util.function.Predicate;

import ru.yandex.direct.intapi.entity.metrika.model.objectinfo.MetrikaTimeToken;
import ru.yandex.direct.intapi.validation.IntApiDefect;
import ru.yandex.direct.validation.Predicates;
import ru.yandex.direct.validation.builder.Constraint;

import static ru.yandex.direct.intapi.entity.metrika.model.objectinfo.MetrikaTimeToken.DATE_TIME_FORMATTER;
import static ru.yandex.direct.intapi.entity.metrika.model.objectinfo.MetrikaTimeToken.SEPARATOR;

public class MetrikaTimeTokenConstraint implements Constraint<String, IntApiDefect> {

    private MetrikaTimeTokenConstraint() {
    }

    public static MetrikaTimeTokenConstraint validTimeToken() {
        return new MetrikaTimeTokenConstraint();
    }

    @Override
    public IntApiDefect apply(String token) {
        Predicate<String> tokenIsValid = t -> {

            /*
                Проверка формата
             */
            int sepIndex = t.indexOf(SEPARATOR);

            // должен присутствовать разделитель
            if (sepIndex < 0) {
                return false;
            }
            // должен быть только один разделитель
            if (sepIndex != t.lastIndexOf(SEPARATOR)) {
                return false;
            }
            // разделитель не должен быть первым или последним символом
            if (sepIndex == 0 || sepIndex == t.length() - 1) {
                return false;
            }

            /*
                Проверка валидности даты
             */
            String dateTime = MetrikaTimeToken.extractRawDateTimeFromTimeToken(t);
            if (!Predicates.validDateTime(DATE_TIME_FORMATTER).test(dateTime)) {
                return false;
            }

            /*
                Проверка валидности id
             */
            String lastId = MetrikaTimeToken.extractRawLastIdFromTimeToken(t);
            if (!Predicates.isLong().test(lastId)) {
                return false;
            }
            if (Long.valueOf(lastId) <= 0) {
                return false;
            }
            return true;
        };

        IntApiDefect defect = IntApiDefect.parameterizedByPath("%s must be <last change time>/<last id> " +
                "(time in ISO LOCAL DATE TIME format like 2017-01-10T23:00:44)");

        return token == null || tokenIsValid.test(token) ? null : defect;
    }
}
