package ru.yandex.direct.intapi.entity.mobilecontent.controller;

import java.net.URI;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import javax.annotation.Nullable;
import javax.ws.rs.core.MediaType;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import one.util.streamex.EntryStream;
import org.apache.commons.lang3.StringUtils;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;

import ru.yandex.direct.core.entity.mobileapp.model.MobileAppAutomaticCreationRequestWithoutSuppliedMobileContent;
import ru.yandex.direct.core.entity.mobileapp.service.MobileAppAutomaticCreationService;
import ru.yandex.direct.core.entity.mobilecontent.container.MobileAppStoreUrl;
import ru.yandex.direct.core.entity.mobilecontent.util.MobileAppStoreUrlParser;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.intapi.ErrorResponse;
import ru.yandex.direct.intapi.IntApiException;
import ru.yandex.direct.intapi.entity.mobilecontent.model.CreateMobileAppRequest;
import ru.yandex.direct.intapi.entity.mobilecontent.model.CreateMobileAppResponse;
import ru.yandex.direct.intapi.entity.mobilecontent.model.MobileAppListResponse;
import ru.yandex.direct.intapi.entity.mobilecontent.model.MobileContentInfoResponse;
import ru.yandex.direct.intapi.entity.mobilecontent.model.SingleMobileAppResponse;
import ru.yandex.direct.intapi.logging.ClientIdParam;
import ru.yandex.direct.web.core.entity.mobilecontent.service.WebCoreMobileAppService;
import ru.yandex.direct.web.core.model.WebErrorResponse;
import ru.yandex.direct.web.core.model.WebResponse;

import static ru.yandex.direct.validation.constraint.StringConstraints.validHref;

@Controller
@RequestMapping(value = "/mobile_app", produces = MediaType.APPLICATION_JSON)
@Api(tags = "mobile_app")
@SuppressWarnings("unused")
public class MobileAppController {
    private final WebCoreMobileAppService webCoreMobileAppService;
    private final MobileAppAutomaticCreationService mobileAppAutomaticCreationService;

    public MobileAppController(
            WebCoreMobileAppService webCoreMobileAppService,
            MobileAppAutomaticCreationService mobileAppAutomaticCreationService) {
        this.webCoreMobileAppService = webCoreMobileAppService;
        this.mobileAppAutomaticCreationService = mobileAppAutomaticCreationService;
    }

    @ApiOperation(
            value = "mobileContentList",
            nickname = "mobileContentList",
            httpMethod = "GET"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 400, message = "Bad params", response = ErrorResponse.class),
                    @ApiResponse(code = 403, message = "permission denied", response = ErrorResponse.class),
                    @ApiResponse(code = 200, message = "Ok", response = MobileAppListResponse.class)
            }
    )
    @RequestMapping(path = "/mobile_content_list", method = RequestMethod.GET)
    @ResponseBody
    public MobileAppListResponse getOldMobileAppList(
            @RequestParam(value = "client_id") @ClientIdParam Long longClientId) {
        ClientId clientId = ClientId.fromLong(longClientId);
        return new MobileAppListResponse(webCoreMobileAppService.getOldAppList(clientId));
    }

    @ApiOperation(
            value = "list",
            nickname = "list",
            httpMethod = "GET"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 400, message = "Bad params", response = ErrorResponse.class),
                    @ApiResponse(code = 403, message = "Permission denied", response = ErrorResponse.class),
                    @ApiResponse(code = 200, message = "Ok", response = MobileAppListResponse.class)
            }
    )
    @RequestMapping(path = "/list", method = RequestMethod.GET)
    @ResponseBody
    public MobileAppListResponse list(
            @RequestParam(value = "client_id") @ClientIdParam Long longClientId,
            @RequestParam(value = "store_content_id", required = false) @Nullable String storeContentId
    ) {
        ClientId clientId = ClientId.fromLong(longClientId);
        return new MobileAppListResponse(webCoreMobileAppService.getAppList(clientId, storeContentId, null));
    }

    @ApiOperation(
            value = "singleApp",
            nickname = "singleApp",
            httpMethod = "GET"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 400, message = "Bad params", response = ErrorResponse.class),
                    @ApiResponse(code = 403, message = "Permission denied", response = ErrorResponse.class),
                    @ApiResponse(code = 200, message = "Ok", response = SingleMobileAppResponse.class)
            }
    )
    @RequestMapping(path = "/singleApp", method = RequestMethod.GET)
    @ResponseBody
    public SingleMobileAppResponse singleApp(@RequestParam(value = "client_id") @ClientIdParam Long longClientId,
                                             @RequestParam(value = "mobile_app_id") Long mobileAppId) {
        ClientId clientId = ClientId.fromLong(longClientId);
        return webCoreMobileAppService.getApp(clientId, mobileAppId)
                .map(SingleMobileAppResponse::new)
                .orElseThrow(() -> new IntApiException(HttpStatus.NOT_FOUND,
                        new ErrorResponse(ErrorResponse.ErrorCode.NOT_FOUND, "Mobile app not found")));
    }

    @ApiOperation(
            value = "createMobileApp",
            nickname = "createMobileApp",
            httpMethod = "POST"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 400, message = "Bad params", response = ErrorResponse.class),
                    @ApiResponse(code = 403, message = "Permission denied", response = ErrorResponse.class),
                    @ApiResponse(code = 200, message = "Ok", response = CreateMobileAppResponse.class)
            }
    )
    @RequestMapping(path = "/createMobileApp", method = RequestMethod.POST)
    @ResponseBody
    public CreateMobileAppResponse createMobileApp(@RequestBody CreateMobileAppRequest request) {
        List<CreateMobileAppRequest.Entity> entities = request.getEntities();

        List<CreateMobileAppResponse.ErrorType> errorsByIndex = new ArrayList<>(entities.size());
        List<MobileAppAutomaticCreationRequestWithoutSuppliedMobileContent> creationRequestsByIndex =
                new ArrayList<>(entities.size());
        List<MobileAppAutomaticCreationRequestWithoutSuppliedMobileContent> validCreationRequests = new ArrayList<>();

        for (CreateMobileAppRequest.Entity entity : entities) {
            String storeHref = entity.getStoreHref();

            Optional<MobileAppStoreUrl> parsedStoreHrefOptional = MobileAppStoreUrlParser.parse(storeHref);

            if (parsedStoreHrefOptional.isEmpty()) {
                creationRequestsByIndex.add(null);
                errorsByIndex.add(CreateMobileAppResponse.ErrorType.INVALID_STORE_HREF);
                continue;
            }

            String trackerUrl = entity.getTrackerUrl();
            if (trackerUrl != null && (StringUtils.isBlank(trackerUrl) || validHref().apply(trackerUrl) != null)) {
                creationRequestsByIndex.add(null);
                errorsByIndex.add(CreateMobileAppResponse.ErrorType.INVALID_TRACKER_URL);
                continue;
            }

            String impressionUrl = entity.getImpressionUrl();
            if (impressionUrl != null &&
                    StringUtils.isNotBlank(impressionUrl) &&
                     validHref().apply(impressionUrl) != null) {
                creationRequestsByIndex.add(null);
                errorsByIndex.add(CreateMobileAppResponse.ErrorType.INVALID_TRACKER_URL);
                continue;
            }

            MobileAppAutomaticCreationRequestWithoutSuppliedMobileContent creationRequest =
                    new MobileAppAutomaticCreationRequestWithoutSuppliedMobileContent(
                            parsedStoreHrefOptional.get(),
                            storeHref,
                            entity.getMinimalOperatingSystemVersion(),
                            trackerUrl,
                            impressionUrl);

            validCreationRequests.add(creationRequest);
            creationRequestsByIndex.add(creationRequest);
            errorsByIndex.add(null);
        }

        Map<MobileAppAutomaticCreationRequestWithoutSuppliedMobileContent, Long> results =
                mobileAppAutomaticCreationService
                        .getOrCreateMobileAppsFromStoreHrefs(request.getClientId(), validCreationRequests);

        List<CreateMobileAppResponse.Entry> resultEntries = EntryStream.zip(creationRequestsByIndex, errorsByIndex)
                .mapKeyValue((req, error) -> {
                    if (error != null) {
                        return CreateMobileAppResponse.Entry.error(error);
                    } else {
                        return CreateMobileAppResponse.Entry.success(results.get(req));
                    }
                })
                .toList();

        return new CreateMobileAppResponse(resultEntries);
    }

    @ApiOperation(
            value = "getMobileContent",
            nickname = "getMobileContent",
            httpMethod = "GET"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 400, message = "Bad params", response = WebErrorResponse.class),
                    @ApiResponse(code = 403, message = "Permission denied", response = WebErrorResponse.class),
                    @ApiResponse(code = 200, message = "Ok", response = MobileContentInfoResponse.class)
            }
    )
    @RequestMapping(path = "/mobile_content", method = RequestMethod.GET)
    @ResponseBody
    public WebResponse getMobileContent(
            @RequestParam(value = "client_id") @ClientIdParam Long longClientId,
            @RequestParam(value = "url") String url) {
        ClientId clientId = ClientId.fromLong(longClientId);
        Optional<MobileAppStoreUrl> parsedUrl = MobileAppStoreUrlParser.parse(url);
        Optional<String> maybeStoreUrl = MobileAppStoreUrlParser.sanitizeUrl(url).map(URI::toString);
        if (!parsedUrl.isPresent() || !maybeStoreUrl.isPresent()) {
            return new WebErrorResponse(400, "Unable to parse URL: " + url);
        }
        return webCoreMobileAppService
                .getMobileContent(clientId, maybeStoreUrl.get(), parsedUrl.get(), /* dryRun = */false)
                .map(r -> (WebResponse) new MobileContentInfoResponse(r))
                .orElse(new WebErrorResponse(400, "Content not found: " + url));
    }

}
