package ru.yandex.direct.intapi.entity.moderation.controller;

import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.ws.rs.core.MediaType;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.intapi.ErrorResponse;
import ru.yandex.direct.intapi.IntApiException;
import ru.yandex.direct.intapi.entity.moderation.model.BannerArchiveStatus;
import ru.yandex.direct.intapi.entity.moderation.model.BannersArchiveStatusRequest;
import ru.yandex.direct.intapi.entity.moderation.model.BlockClientResult;
import ru.yandex.direct.intapi.entity.moderation.model.BlockClientsRequest;
import ru.yandex.direct.intapi.entity.moderation.model.BlockClientsResponse;
import ru.yandex.direct.intapi.entity.moderation.model.CidAndDomainInfoResponse;
import ru.yandex.direct.intapi.entity.moderation.model.ModerateCampaignsRequest;
import ru.yandex.direct.intapi.entity.moderation.model.ModerateCampaignsResponse;
import ru.yandex.direct.intapi.entity.moderation.model.UnblockClientResult;
import ru.yandex.direct.intapi.entity.moderation.model.UnblockClientsRequest;
import ru.yandex.direct.intapi.entity.moderation.model.UnblockClientsResponse;
import ru.yandex.direct.intapi.entity.moderation.model.client.ModerationClientParams;
import ru.yandex.direct.intapi.entity.moderation.model.client.ModerationClientsRequest;
import ru.yandex.direct.intapi.entity.moderation.model.client.ModerationClientsResponse;
import ru.yandex.direct.intapi.entity.moderation.model.modedit.ModerationEditObjectsResult;
import ru.yandex.direct.intapi.entity.moderation.model.modedit.ModerationEditReplacement;
import ru.yandex.direct.intapi.entity.moderation.model.modresync.ImportToResyncQueueRequest;
import ru.yandex.direct.intapi.entity.moderation.service.IntapiModerationService;
import ru.yandex.direct.intapi.entity.moderation.service.ModResyncQueueImportService;
import ru.yandex.direct.intapi.entity.moderation.service.block.ModerationBlockService;
import ru.yandex.direct.intapi.entity.moderation.service.client.ModerationClientService;
import ru.yandex.direct.intapi.entity.moderation.service.modedit.ModerationEditService;
import ru.yandex.direct.intapi.validation.kernel.ValidationResultConversionService;
import ru.yandex.direct.intapi.validation.model.IntapiValidationResponse;
import ru.yandex.direct.intapi.webapp.semaphore.Semaphore;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.traceinterception.entity.traceinterception.exception.TraceInterceptionSemaphoreException;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;
import ru.yandex.direct.tvm.AllowServices;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.web.core.model.WebErrorResponse;
import ru.yandex.direct.web.core.model.WebResponse;

import static java.util.stream.Collectors.joining;
import static ru.yandex.direct.intapi.entity.moderation.service.IntapiModerationService.CLIENTS_BANNED_IN_PERIOD_LIMIT;
import static ru.yandex.direct.tvm.TvmService.CLIENT_BAN_PROD;
import static ru.yandex.direct.tvm.TvmService.CLIENT_BAN_TEST;
import static ru.yandex.direct.tvm.TvmService.DIRECT_DEVELOPER;
import static ru.yandex.direct.tvm.TvmService.FURY_SUPERMOD_PREPROD;
import static ru.yandex.direct.tvm.TvmService.FURY_SUPERMOD_PROD;
import static ru.yandex.direct.tvm.TvmService.FURY_SUPERMOD_TEST;
import static ru.yandex.direct.tvm.TvmService.MODADVERT_MANUAL_PROD;
import static ru.yandex.direct.tvm.TvmService.MODQUALITY_NIRVANA_ROBOT;
import static ru.yandex.direct.tvm.TvmService.SUPERMODERATION_PROD;
import static ru.yandex.direct.tvm.TvmService.SUPERMODERATION_TEST;
import static ru.yandex.direct.tvm.TvmService.YABS_DIRECT_MONITORING;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@RestController
@Api(tags = "moderation")
@RequestMapping("/moderation")
@ParametersAreNonnullByDefault
public class ModerationController {
    private static final Logger logger = LoggerFactory.getLogger(ModerationController.class);

    private static final int MAX_CLIENT_IDS_LIMIT = 300;
    private static final int MAX_BANNER_IDS_LIMIT = 1000;

    private static final String REMODERATION_SEMAPHORE_KEY = "remoderation_semaphore";
    private static final int REMODERATION_SEMAPHORE_PERMITS = 3;

    private static final String MODERATION_EDIT_SEMAPHORE_KEY = "moderation_edit_semaphore";
    private static final int MODERATION_EDIT_SEMAPHORE_PERMITS = 3;

    private static final String MODERATION_CLIENTS_SEMAPHORE_KEY = "moderation_clients_semaphore";
    private static final int MODERATION_CLIENTS_SEMAPHORE_PERMITS = 3;

    private final IntapiModerationService intapiModerationService;
    private final ModResyncQueueImportService modResyncQueueImportService;
    private final ModerationEditService moderationEditService;
    private final ValidationResultConversionService conversionService;
    private final ModerationClientService moderationClientService;
    private final ModerationBlockService moderationBlockService;

    @Autowired
    public ModerationController(IntapiModerationService intapiModerationService,
                                ModResyncQueueImportService modResyncQueueImportService,
                                ModerationEditService moderationEditService,
                                ValidationResultConversionService conversionService,
                                ModerationClientService moderationClientService,
                                ModerationBlockService moderationBlockService) {
        this.intapiModerationService = intapiModerationService;
        this.modResyncQueueImportService = modResyncQueueImportService;
        this.moderationEditService = moderationEditService;
        this.conversionService = conversionService;
        this.moderationClientService = moderationClientService;
        this.moderationBlockService = moderationBlockService;
    }

    @ApiOperation(
            value = "Проставить статусы модерации кампаниям с agencyId из проперти " +
                    "bs_agency_ids_for_direct_monitoring. Сейчас там указано агентство bkagentstvo.",
            httpMethod = "POST",
            nickname = "moderate_campaigns"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 403, message = "Permission denied", response = WebErrorResponse.class),
                    @ApiResponse(code = 200, message = "Ok", response = ModerateCampaignsResponse.class)
            }
    )
    @RequestMapping(path = "/moderate_campaigns", method = RequestMethod.POST)
    @AllowServices(production = {YABS_DIRECT_MONITORING}, testing = {YABS_DIRECT_MONITORING, DIRECT_DEVELOPER})
    public ModerateCampaignsResponse moderateCampaigns(
            @ApiParam(value = "ID кампаний") @RequestBody ModerateCampaignsRequest data) {
        List<Long> moderatedCampaignIds = intapiModerationService.moderateCampaigns(data.getCampaignIds());
        return new ModerateCampaignsResponse(moderatedCampaignIds);
    }

    @ApiOperation(
            value = "Выдаёт для каждого переданного bannerId статус архивности с учётом статуса архивности кампании",
            httpMethod = "POST",
            nickname = "banners_archive_status"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 400, message = "Некорректный запрос", response = ErrorResponse.class),
                    @ApiResponse(code = 403, message = "Permission denied", response = WebErrorResponse.class),
                    @ApiResponse(code = 200, message = "Ok", response = List.class)
            }
    )
    @RequestMapping(path = "/banners_archive_status", method = RequestMethod.POST)
    @AllowServices(production = {SUPERMODERATION_PROD}, testing = {SUPERMODERATION_TEST, DIRECT_DEVELOPER})
    public List<BannerArchiveStatus> getBannersArchiveStatus(
            @ApiParam(value = "ID баннеров") @RequestBody BannersArchiveStatusRequest request) {
        if (request.getBannerIds().size() > MAX_BANNER_IDS_LIMIT) {
            throw new IntApiException(
                    HttpStatus.BAD_REQUEST,
                    new ErrorResponse(
                            ErrorResponse.ErrorCode.BAD_PARAM,
                            String.format("banner_ids size limit %d exceeded", MAX_BANNER_IDS_LIMIT)));
        }
        var bannersArchiveStatus = intapiModerationService.getBannersArchiveStatus(request.getBannerIds());
        return EntryStream.of(bannersArchiveStatus)
                .mapKeyValue(BannerArchiveStatus::new)
                .toList();
    }


    @ApiOperation(
            value = "Получает список пар (cid, домен), а также флаги и минус регионы по клиентскому uid/login",
            nickname = "cid_and_domain_info",
            httpMethod = "GET"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 400, message = "Некорректный запрос", response = ErrorResponse.class),
                    @ApiResponse(code = 403, message = "Permission denied", response = WebErrorResponse.class),
                    @ApiResponse(code = 200, message = "Ok", response = CidAndDomainInfoResponse.class)
            }
    )
    @RequestMapping(path = "/cid_and_domain_info", method = RequestMethod.GET)
    @AllowServices(production = {MODQUALITY_NIRVANA_ROBOT}, testing = {MODQUALITY_NIRVANA_ROBOT, DIRECT_DEVELOPER})
    public WebResponse getCampaignsAndBannerInfoByUser(
            @RequestParam(required = false) Long uid,
            @RequestParam(required = false) String login,
            @RequestParam(required = false) @Nullable Integer offset,
            @RequestParam(required = false, value = "row_number") @Nullable Integer rowNumber
    ) {
        return intapiModerationService.getCidAndDomainInfoByUid(uid, login, offset, rowNumber);
    }

    @ApiOperation(
            value = "Блокировка пользователей по clientIds. Вответ в формате JSON с данными " +
                    "о предыдущем состоянии блокировки",
            nickname = "/block_clients"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 400, message = "Некорректный запрос или ошибка валидации", response =
                            ErrorResponse.class),
                    @ApiResponse(code = 403, message = "Permission denied", response = WebErrorResponse.class),
                    @ApiResponse(code = 200, message = "Ok", response = BlockClientsResponse.class)
            }
    )
    @RequestMapping(path = "/block_clients", method = RequestMethod.POST, produces = MediaType.APPLICATION_JSON)
    @AllowServices(
            production = {SUPERMODERATION_PROD, FURY_SUPERMOD_PROD, CLIENT_BAN_PROD},
            testing = {SUPERMODERATION_TEST, FURY_SUPERMOD_PREPROD, FURY_SUPERMOD_TEST,
                CLIENT_BAN_TEST, DIRECT_DEVELOPER})
    public BlockClientsResponse blockClients(
            @ApiParam(value = "ID клиентов")
            @RequestBody @Nonnull BlockClientsRequest data) {
        List<ClientId> clientIds = StreamEx.of(data.getClientIds())
                .distinct()
                .map(ClientId::fromLong)
                .toImmutableList();
        if (clientIds.size() > MAX_CLIENT_IDS_LIMIT) {
            throw new IntApiException(
                    HttpStatus.BAD_REQUEST,
                    new ErrorResponse(
                            ErrorResponse.ErrorCode.BAD_PARAM,
                            String.format("Too many clients in one call. Limit is %d clients", MAX_CLIENT_IDS_LIMIT)));
        }
        if (data.getBlockComment() != null && data.getBlockComment().length() > 65_535) {
            throw new IntApiException(
                    HttpStatus.BAD_REQUEST,
                    new ErrorResponse(ErrorResponse.ErrorCode.BAD_PARAM, "blockComment max size exceeded"));
        }

        // Дополнительные проверки на "разумность"
        // Можно их отключить, передав специальный флаг
        if (!data.isIgnoreSanityChecks()) {
            // Получаем важных клиентов из YT
            // Если Hahn недоступен -- ручка будет отвечать ошибкой, решили с Модерацией, что это ок
            // Если нужно будет заблокировать в это время -- можно воспользоваться force-флагом
            Set<ClientId> importantClients;
            // Трейсинг -- на него можно повесить семафор на всякий случай
            // (чтобы антифрод нас не заддосил случайно)
            // Условие для интерсептора (в trace_interceptions)
            // {"method": "moderation.block_clients", "service": "direct.intapi", "func": "getImportantClients"}
            try (TraceProfile traceProfile = Trace.current().profile("getImportantClients")) {
                importantClients = intapiModerationService.getImportantClients(clientIds);
            } catch (TraceInterceptionSemaphoreException e) {
                logger.error("Can't get important clients", e);
                throw new IntApiException(
                        HttpStatus.BAD_REQUEST,
                        new ErrorResponse(
                                ErrorResponse.ErrorCode.TOO_MANY_REQUESTS,
                                "Unable to retrieve important clients from YT table: too many simultaneous requests"));
            } catch (RuntimeException e) {
                logger.error("Can't get important clients", e);
                throw new IntApiException(
                        HttpStatus.BAD_REQUEST,
                        new ErrorResponse(
                                ErrorResponse.ErrorCode.INTERNAL_ERROR,
                                "Unable to retrieve important clients from YT table"));
            }
            if (!importantClients.isEmpty()) {
                String errorMessage = String.format("Unable to block important clients: %s",
                        importantClients.stream().map(ClientId::toString).collect(joining(",")));
                logger.error("{}", errorMessage);
                throw new IntApiException(HttpStatus.BAD_REQUEST,
                        new ErrorResponse(ErrorResponse.ErrorCode.BAD_PARAM, errorMessage));
            }

            // Атомарно пытаемся записать в ppc_property блокируемых клиентов
            // Если за последние 24 часа туда уже было записано больше допустимого, то выходим с ошибкой
            if (!intapiModerationService.reserveClientsForBlocking(clientIds)) {
                throw new IntApiException(
                        HttpStatus.BAD_REQUEST,
                        new ErrorResponse(
                                ErrorResponse.ErrorCode.TOO_MANY_REQUESTS,
                                String.format("Too many clients have been blocked in 24 hrs. Current limit is %d",
                                        CLIENTS_BANNED_IN_PERIOD_LIMIT)));
            }
        }

        Map<ClientId, BlockClientResult> results = moderationBlockService.blockClients(
                clientIds,
                data.getBlockReasonType(),
                data.getBlockComment());

        return new BlockClientsResponse(mapList(data.getClientIds(), id -> results.get(ClientId.fromLong(id))));
    }

    @ApiOperation(
            value = "Разблокировка пользователей по clientId",
            nickname = "/unblock_clients"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 400, message = "Некорректный запрос или ошибка валидации", response =
                            ErrorResponse.class),
                    @ApiResponse(code = 403, message = "Permission denied", response = WebErrorResponse.class),
                    @ApiResponse(code = 200, message = "Ok", response = UnblockClientsResponse.class)
            }
    )
    @RequestMapping(path = "/unblock_clients", method = RequestMethod.POST, produces = MediaType.APPLICATION_JSON)
    @AllowServices(
            production = {SUPERMODERATION_PROD, FURY_SUPERMOD_PROD, CLIENT_BAN_PROD},
            testing = {SUPERMODERATION_TEST, FURY_SUPERMOD_PREPROD, FURY_SUPERMOD_TEST,
                    CLIENT_BAN_TEST, DIRECT_DEVELOPER}
    )
    public UnblockClientsResponse unblockClients(
            @ApiParam(value = "ID клиентов")
            @RequestBody @Nonnull UnblockClientsRequest data) {
        List<ClientId> clientIds = StreamEx.of(data.getClientIds())
                .distinct()
                .map(ClientId::fromLong)
                .toImmutableList();
        if (clientIds.size() > MAX_CLIENT_IDS_LIMIT) {
            throw new IntApiException(
                    HttpStatus.BAD_REQUEST,
                    new ErrorResponse(
                            ErrorResponse.ErrorCode.BAD_PARAM,
                            String.format("Too many clients in one call. Limit is %d clients", MAX_CLIENT_IDS_LIMIT)));
        }

        Map<ClientId, UnblockClientResult> results = moderationBlockService.unblockClients(clientIds);

        return new UnblockClientsResponse(mapList(data.getClientIds(), id -> results.get(ClientId.fromLong(id))));
    }

    @ApiOperation(
            value = "Добавление объектов в ppc.mod_resync_queue, " +
                    "аналог Perl-овой страницы в админке importIntoModerationResyncQueue",
            nickname = "/mod_resync_queue_import"
    )
    @RequestMapping(path = "/mod_resync_queue_import",
            method = RequestMethod.POST, produces = MediaType.APPLICATION_JSON)
    @Semaphore(key = REMODERATION_SEMAPHORE_KEY,
            permits = REMODERATION_SEMAPHORE_PERMITS,
            retryTimeout = 3_000)
    @AllowServices(production = {SUPERMODERATION_PROD}, testing = {SUPERMODERATION_TEST, DIRECT_DEVELOPER})
    public IntapiValidationResponse importToModResyncQueue(
            @ApiParam(value = "Описание объектов для постановки в очередь")
            @RequestBody @Nonnull ImportToResyncQueueRequest request) {
        Result<Integer> result = modResyncQueueImportService.importToModResyncQueue(request);
        return conversionService.buildValidationResponse(result);
    }

    @ApiOperation(
            value = "Исправление опечаток в текстах баннеров и ассетов " +
                    "(временная ручка до полного перехода на новый транспорт с Модерацией)",
            nickname = "/mod_edit"
    )
    @RequestMapping(path = "/mod_edit",
            method = RequestMethod.POST, produces = MediaType.APPLICATION_JSON)
    @Semaphore(key = MODERATION_EDIT_SEMAPHORE_KEY,
            permits = MODERATION_EDIT_SEMAPHORE_PERMITS,
            retryTimeout = 3_000)
    @AllowServices(production = {MODADVERT_MANUAL_PROD}, testing = {SUPERMODERATION_TEST, DIRECT_DEVELOPER})
    public ModerationEditObjectsResult moderationEdit(
            @ApiParam(value = "Описание изменений: какие старые тексты нужно заменить на новые")
            @RequestBody @Nonnull List<ModerationEditReplacement> replacements) {
        return moderationEditService.editObjects(replacements);
    }

    @ApiOperation(
            value = "Получение параметров клиентов",
            nickname = "/clients"
    )
    @RequestMapping(path = "/clients",
            method = RequestMethod.POST, produces = MediaType.APPLICATION_JSON)
    @Semaphore(key = MODERATION_CLIENTS_SEMAPHORE_KEY,
            permits = MODERATION_CLIENTS_SEMAPHORE_PERMITS,
            retryTimeout = 3_000)
    @AllowServices(production = {MODADVERT_MANUAL_PROD}, testing = {SUPERMODERATION_TEST, DIRECT_DEVELOPER})
    public ModerationClientsResponse getClientParams(
            @ApiParam(value = "Условие фильтрации клиентов")
            @RequestBody @Nonnull ModerationClientsRequest request) {

        ValidationResult<ModerationClientsRequest, Defect> validationResult =
                moderationClientService.validateClientsRequest(request);
        if (validationResult.hasAnyErrors()) {
            return new ModerationClientsResponse(conversionService.buildIntapiValidationResult(validationResult));
        }

        Map<Long, ModerationClientParams> clientParamsMap =
                moderationClientService.getClientParams(request.getClientIds());
        return new ModerationClientsResponse(clientParamsMap);
    }
}
