package ru.yandex.direct.intapi.entity.moderation.service;

import java.time.Duration;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.core.type.TypeReference;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.collections4.ListUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.common.db.PpcPropertyNames;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.banner.model.BannerStatusModerate;
import ru.yandex.direct.core.entity.banner.model.TextBanner;
import ru.yandex.direct.core.entity.banner.repository.BannerModerationRepository;
import ru.yandex.direct.core.entity.banner.repository.BannerRelationsRepository;
import ru.yandex.direct.core.entity.bids.model.Bid;
import ru.yandex.direct.core.entity.bids.repository.BidRepository;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.core.entity.user.service.validation.BlockUserValidationService;
import ru.yandex.direct.dbschema.ppc.enums.BannersStatuspostmoderate;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.intapi.entity.moderation.model.BannedClient;
import ru.yandex.direct.intapi.entity.moderation.model.CidAndDomainInfo;
import ru.yandex.direct.intapi.entity.moderation.model.CidAndDomainInfoResponse;
import ru.yandex.direct.web.core.model.WebErrorResponse;
import ru.yandex.direct.web.core.model.WebResponse;
import ru.yandex.misc.lang.StringUtils;

import static com.google.common.base.MoreObjects.firstNonNull;
import static java.time.temporal.ChronoUnit.MILLIS;
import static java.util.Collections.emptySet;
import static java.util.function.Function.identity;
import static java.util.stream.Collectors.toMap;
import static java.util.stream.Collectors.toSet;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static ru.yandex.direct.common.db.PpcPropertyNames.BANNED_CLIENTS;
import static ru.yandex.direct.utils.FunctionalUtils.flatMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.JsonUtils.fromJson;
import static ru.yandex.direct.utils.JsonUtils.toJson;
import static ru.yandex.direct.utils.ThreadUtils.sleep;

@Service
@ParametersAreNonnullByDefault
public class IntapiModerationService {
    private static final Logger logger = LoggerFactory.getLogger(IntapiModerationService.class);

    /**
     * Разрешаем блокировать не более 700 клиентов в сутки
     * Последний раз увеличение лимита было тут: https://st.yandex-team.ru/DIRECT-161939#621360aba5bebb2042603e3f
     */
    public static final int CLIENTS_BANNED_IN_PERIOD_LIMIT = 700;
    private static final int CAMPAIGNS_CHUNK_SIZE = 200;

    private final ShardHelper shardHelper;
    private final CampaignRepository campaignRepository;
    private final AdGroupRepository adGroupRepository;
    private final BannerModerationRepository bannerModerationRepository;
    private final BannerRelationsRepository bannerRelationsRepository;
    private final BidRepository bidRepository;
    private final UserService userService;
    private final BlockUserValidationService blockUserValidationService;
    private final PpcProperty<Set<Long>> directMonitoringAgencyIds;
    private final PpcPropertiesSupport ppcPropertiesSupport;

    @Autowired
    @SuppressWarnings("ParameterNumber")
    public IntapiModerationService(
            ShardHelper shardHelper,
            PpcPropertiesSupport ppcPropertiesSupport,
            CampaignRepository campaignRepository,
            AdGroupRepository adGroupRepository,
            BannerModerationRepository bannerModerationRepository,
            BannerRelationsRepository bannerRelationsRepository,
            BidRepository bidRepository,
            UserService userService,
            BlockUserValidationService blockUserValidationService) {
        this.shardHelper = shardHelper;
        this.campaignRepository = campaignRepository;
        this.adGroupRepository = adGroupRepository;
        this.bannerModerationRepository = bannerModerationRepository;
        this.bannerRelationsRepository = bannerRelationsRepository;
        this.bidRepository = bidRepository;
        this.userService = userService;
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.blockUserValidationService = blockUserValidationService;

        this.directMonitoringAgencyIds = ppcPropertiesSupport.get(
                PpcPropertyNames.BS_AGENCY_IDS_FOR_DIRECT_MONITORING, Duration.ofMinutes(1));
    }

    /**
     * DIRECT-114748
     * Метод для проставления положительных статусов модерации рекламным объектам, используемым в Директ-Мониторинге.
     * Принимает список id кампаний, которым нужно проставить статусы модерации. Выбирает из них те, которые имеют
     * тип TEXT и agencyId которых содержится в проперте
     * {@link PpcPropertyNames#BS_AGENCY_IDS_FOR_DIRECT_MONITORING}.
     * Выбранным кампаниям проставляет statusModerate = "Yes", statusPostModerate = "Accepted";
     * группам из этих кампаний проставляет statusModerate = "Yes", statusPostModerate = "Yes";
     * баннерам из этих кампаний проставляет statusModerate = "Yes", statusPostModerate = "Yes";
     * ключевым словам из этих кампаний проставляет statusModerate = "Yes"
     *
     * @param campaignIds Список id кампаний, которым нужно проставить статусы модерации.
     * @return Список id кампаний, которым проставились статусы модерации.
     */
    public List<Long> moderateCampaigns(Collection<Long> campaignIds) {
        List<Long> moderatedCampaignIds = new ArrayList<>();
        shardHelper.groupByShard(campaignIds, ShardKey.CID)
                .forEach((shard, cids) -> moderatedCampaignIds.addAll(moderateCampaigns(shard, cids)));
        return moderatedCampaignIds;
    }

    private List<Long> moderateCampaigns(int shard, Collection<Long> campaignIds) {
        List<Long> filteredCampaignIds = StreamEx.of(campaignRepository.getCampaigns(shard, campaignIds))
                .filter(c -> c.getType().equals(CampaignType.TEXT))
                .filter(c -> directMonitoringAgencyIds.getOrDefault(emptySet()).contains(c.getAgencyId()))
                .map(Campaign::getId)
                .toList();
        campaignRepository.markCampaignsAsModerated(shard, filteredCampaignIds);

        List<Long> adGroupIds = adGroupRepository.getAdGroupIdsByCampaignIds(shard, filteredCampaignIds)
                .values().stream()
                .flatMap(List::stream)
                .collect(Collectors.toList());
        adGroupRepository.markAdGroupsAsModerated(shard, adGroupIds);

        List<Long> bannerIds = bannerRelationsRepository.getBannerIdsByCampaignIdsAndBannerTypes(
                shard, filteredCampaignIds, List.of(TextBanner.class));
        bannerModerationRepository.updateStatusModerateAndPostModerate(
                shard, bannerIds, BannerStatusModerate.YES, BannersStatuspostmoderate.Yes);

        List<Long> bidIds = bidRepository.getBidsByCampaignIds(shard, filteredCampaignIds)
                .stream().map(Bid::getId).collect(Collectors.toList());
        bidRepository.markBidsAsModerated(shard, bidIds);

        return filteredCampaignIds;
    }

    public Map<ClientId, Boolean> getCurrentStatusBlock(Collection<User> users) {
        return users.stream().collect(toMap(User::getClientId,
                User::getStatusBlocked, (isUserBlocked1, isUserBlocked2) -> isUserBlocked1 && isUserBlocked2));
    }

    public List<User> getUsersByClientIds(List<ClientId> clientIds) {
        Map<ClientId, List<Long>> clients = userService.massGetUidsByClientIds(clientIds);
        return mapList(userService.massGetUser(flatMap(clients.values(), identity())), identity());
    }

    public Map<Long, Boolean> getBannersArchiveStatus(Collection<Long> bannerIds) {
        if (bannerIds.isEmpty()) {
            return Collections.emptyMap();
        }
        Map<Long, Boolean> result = new HashMap<>();
        shardHelper.groupByShard(bannerIds, ShardKey.BID).forEach((shard, ids) ->
                result.putAll(bannerModerationRepository.getBannersArchiveStatus(shard, ids))
        );
        return result;
    }

    /**
     * Получает по uid/login все пары cid и доменов баннеров для данного cid,
     * для каждой пары группируются флаги и минус гео.
     *
     * @param offset Начало
     * @param rowNumber Количество строк
     */
    public WebResponse getCidAndDomainInfoByUid(@Nullable Long uid,
                                                @Nullable String login,
                                                @Nullable Integer offset,
                                                @Nullable Integer rowNumber) {
        ClientId clientId;
        try {
            clientId = uid != null
                    ? ClientId.fromLong(shardHelper.getClientIdByUid(uid))
                    : ClientId.fromLong(shardHelper.getClientIdByLogin(login));
        } catch (IllegalArgumentException e) {
            return new WebErrorResponse(HttpStatus.BAD_REQUEST.value(), e.getMessage());
        }

        var shard = shardHelper.getShardByClientId(clientId);
        List<Long> campaignIds = campaignRepository.getSortedNonArchivedCampaignIdsByClientId(shard, clientId);

        var moderationInfos = ListUtils.partition(campaignIds, CAMPAIGNS_CHUNK_SIZE)
                .stream()
                .flatMap(cids -> {
                    var cidAndBannerInfos = bannerModerationRepository.getCidAndBannersInfo(shard, cids);
                    var minusGeoInfos = bannerModerationRepository.getDomainAndMinusGeoByCampaignIds(shard, cids);

                    return EntryStream.of(cidAndBannerInfos)
                            .map((entry) -> {
                                var cidAndDomain = entry.getKey();
                                return new CidAndDomainInfo(
                                        cidAndDomain.getCid(),
                                        StringUtils.reverse(cidAndDomain.getDomain()),
                                        entry.getValue(),
                                        parseMinusGeoValuesByType(minusGeoInfos.getOrDefault(cidAndDomain, "")));
                            })
                            .sorted(Comparator.comparing(CidAndDomainInfo::getCid)
                                    .thenComparing(CidAndDomainInfo::getDomain));
                });

        if (offset != null) {
            moderationInfos = moderationInfos.skip(offset);
        }
        if (rowNumber != null) {
            moderationInfos = moderationInfos.limit(rowNumber);
        }
        return new CidAndDomainInfoResponse(moderationInfos.collect(Collectors.toList()));
    }

    private Map<String, Set<Long>> parseMinusGeoValuesByType(String record) {
        Map<String, Set<Long>> resultingMap = new HashMap<>();
        if (isNotBlank(record)) {
            StreamEx.of(record.split(" "))
                    .map(el -> el.split(":"))
                    .forEach(el -> {
                        var key = el[0];
                        var entries = Arrays.stream(el[1].split(","))
                                .map(Long::valueOf)
                                .collect(Collectors.toSet());
                        var entrySet = resultingMap.getOrDefault(key, new HashSet<>());
                        entrySet.addAll(entries);

                        resultingMap.put(key, entrySet);
                    });
        }

        return resultingMap;
    }

    public boolean reserveClientsForBlocking(List<ClientId> clientIds) {
        boolean changed;
        int unsuccessfulAttempts = 0;
        do {
            String oldState = ppcPropertiesSupport.get(BANNED_CLIENTS.getName());
            List<BannedClient> oldBannedClients = fromJson(firstNonNull(oldState, "[]"), new TypeReference<>() {
            });
            Instant now = Instant.now();
            Instant yesterday = now.minus(24, ChronoUnit.HOURS);
            Set<Long> clientIdsSet = clientIds.stream().map(ClientId::asLong).collect(toSet());
            List<BannedClient> bannedTodayBefore = StreamEx.of(oldBannedClients)
                    .filter(bannedClient -> bannedClient.getTimestamp().isAfter(yesterday))
                    .toList();
            List<BannedClient> bannedTodayAfter = StreamEx.of(bannedTodayBefore)
                    // Исключаем клиентов, которых собираемся заблокировать в текущем запросе
                    // (антифрод может прислать одного клиента несколько раз, и это не должно
                    // приводить к исчерпанию лимита)
                    .filter(bannedClient -> !clientIdsSet.contains(bannedClient.getClientId()))
                    .append(mapList(clientIds, clientId -> new BannedClient(clientId.asLong(), now)))
                    .toList();
            if (bannedTodayAfter.size() > CLIENTS_BANNED_IN_PERIOD_LIMIT) {
                logger.error("Can't reserve clients for blocking: limit exceeded. Clients banned today: {}",
                        bannedTodayBefore);
                return false;
            }
            String newState = toJson(bannedTodayAfter);

            changed = ppcPropertiesSupport.cas(BANNED_CLIENTS.getName(), oldState, newState);
            if (!changed) {
                logger.warn("{} property value was not changed", BANNED_CLIENTS.getName());
                unsuccessfulAttempts += 1;
                sleep(500, MILLIS);
            }
        } while (!changed && unsuccessfulAttempts <= 10);
        return changed;
    }

    public Set<ClientId> getImportantClients(List<ClientId> clientIds) {
        return blockUserValidationService.getImportantClients(clientIds);
    }
}
