package ru.yandex.direct.intapi.entity.moderation.service;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.BiFunction;

import com.google.common.collect.Lists;
import one.util.streamex.EntryStream;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.moderation.model.ModResyncQueueObj;
import ru.yandex.direct.core.entity.moderation.model.ModResyncQueueObjectType;
import ru.yandex.direct.core.entity.moderation.repository.ModResyncQueueRepository;
import ru.yandex.direct.dbutil.QueryWithForbiddenShardMapping;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.intapi.entity.moderation.model.modresync.ImportToResyncQueueRequest;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS_PERFORMANCE;
import static ru.yandex.direct.dbschema.ppc.tables.AdditionsItemCallouts.ADDITIONS_ITEM_CALLOUTS;
import static ru.yandex.direct.dbschema.ppc.tables.MobileContent.MOBILE_CONTENT;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.collectionSize;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;

@Service
public class ModResyncQueueImportService {

    private static final Integer IDS_LIMIT = 50_000;
    private static final Integer DB_REQUEST_LIMIT = 1_000;

    // Выставляем высокий приоритет, чтоб записи на перемодерацию из ручки извлекались до
    // записей, которые делает moderateClientNew для скопированных кампаний:
    // https://a.yandex-team.ru/arcadia/direct/perl/protected/Moderate/Export.pm?rev=r9585517#L329
    private static final long DEFAULT_PRIORITY = 110L;
    private static final boolean DEFAULT_REMODERATE = false;

    private final DslContextProvider dslContextProvider;
    private final ShardHelper shardHelper;
    private final ModResyncQueueRepository modResyncQueueRepository;

    @Autowired
    public ModResyncQueueImportService(DslContextProvider dslContextProvider, ShardHelper shardHelper,
                                       ModResyncQueueRepository modResyncQueueRepository) {
        this.dslContextProvider = dslContextProvider;
        this.shardHelper = shardHelper;
        this.modResyncQueueRepository = modResyncQueueRepository;
    }

    public Result<Integer> importToModResyncQueue(ImportToResyncQueueRequest request) {
        ValidationResult<ImportToResyncQueueRequest, Defect> vr = validate(request);
        if (vr.hasAnyErrors()) {
            return Result.broken(vr);
        }

        Map<Integer, List<Long>> shardToIdsMap = groupByShards(request.getType(), request.getIds());
        EntryStream.of(shardToIdsMap).forKeyValue((shard, ids) -> {
            Lists.partition(ids, DB_REQUEST_LIMIT).forEach(idsChunk -> {
                List<Long> objectIds = convertIdsToModResyncQueueObjectIds(shard, request.getType(), idsChunk);
                List<ModResyncQueueObj> modResyncQueueObjects =
                        convertIdsToObjects(request.getType(), objectIds, DEFAULT_PRIORITY, DEFAULT_REMODERATE);
                modResyncQueueRepository.add(shard, modResyncQueueObjects);
            });
        });

        return Result.successful(request.getIds().size());
    }

    private ValidationResult<ImportToResyncQueueRequest, Defect> validate(ImportToResyncQueueRequest request) {
        ItemValidationBuilder<ImportToResyncQueueRequest, Defect> vb = ItemValidationBuilder.of(request);

        vb.item(request.getType(), ImportToResyncQueueRequest.TYPE)
                .check(notNull());

        vb.list(request.getIds(), ImportToResyncQueueRequest.IDS)
                .check(notNull())
                .check(collectionSize(1, IDS_LIMIT))
                .checkEach(notNull())
                .checkEach(validId());

        return vb.getResult();
    }

    private Map<Integer, List<Long>> groupByShards(ModResyncQueueObjectType type, List<Long> ids) {
        switch (type) {
            case BANNER:
            case CONTACTINFO:
            case SITELINKS_SET:
            case IMAGE:
            case DISPLAY_HREF:
            case TURBOLANDING:
            case IMAGE_AD:
            case CANVAS:
            case VIDEO_ADDITION:
            case HTML5_CREATIVE:
            case CPM_VIDEO:
            case CPM_YNDX_FRONTPAGE:
            case FIXCPM_YNDX_FRONTPAGE:
            case BANNER_BUTTON:
            case BANNER_LOGO:
            case PERF_CREATIVE:
            case BANNER_MULTICARD:
                return groupBannersByShards(ids);
            case PHRASES:
                return groupAdGroupsByShards(ids);
            case MOBILE_CONTENT:
                return groupObjectsByShardsBruteForce(ids, this::getMobileContentIds);
            case CALLOUT:
                return groupObjectsByShardsBruteForce(ids, this::getCalloutIds);
            default:
                throw new IllegalStateException("unsupported type: " + type);
        }
    }

    /**
     * Конвертирует ids, переданные в ручку, в идентификаторы объектов, которые ожидаются в таблице mod_resync_queue
     */
    private List<Long> convertIdsToModResyncQueueObjectIds(int shard, ModResyncQueueObjectType type, List<Long> ids) {
        switch (type) {
            // TODO : эти объекты отправляются старым транспортом при изменении statusModerate в banners_performance
            //  а новый транспорт отправляет их при измененнии statusModerate в banners
            //  когда эти типы будут переключены на новый транспорт, здесь (и в перле) нужно будет поменять на bid
            case CANVAS:
            case VIDEO_ADDITION:
            case HTML5_CREATIVE:
                return getBannerCreativeIds(shard, ids);
            case PERF_CREATIVE:
                return getCreativeIds(shard, ids);
            default:
                return ids;
        }
    }

    @QueryWithForbiddenShardMapping("в ручку приходят только id объектов без шарда (вызывается единицы раз в день)")
    private Map<Integer, List<Long>> groupBannersByShards(List<Long> ids) {
        return shardHelper.groupByShard(ids, ShardKey.BID).getShardedDataMap();
    }

    @QueryWithForbiddenShardMapping("в ручку приходят только id объектов без шарда (вызывается единицы раз в день)")
    private Map<Integer, List<Long>> groupAdGroupsByShards(List<Long> ids) {
        return shardHelper.groupByShard(ids, ShardKey.PID).getShardedDataMap();
    }

    // ручка вызывается примерно единицы раз в день, поэтому такого перебора не боимся
    private Map<Integer, List<Long>> groupObjectsByShardsBruteForce(
            List<Long> ids,
            BiFunction<Integer, Collection<Long>, List<Long>> selectFunction) {
        Map<Integer, List<Long>> shardedDataMap = new HashMap<>();
        shardHelper.forEachShard(shard -> {
            Lists.partition(ids, DB_REQUEST_LIMIT).forEach(idsChunk -> {
                List<Long> idsInCurrentShard = selectFunction.apply(shard, idsChunk);
                shardedDataMap.computeIfAbsent(shard, s -> new ArrayList<>()).addAll(idsInCurrentShard);
            });
        });
        return shardedDataMap;
    }

    private List<Long> getBannerCreativeIds(int shard, Collection<Long> bannerIds) {
        return dslContextProvider.ppc(shard)
                .select(BANNERS_PERFORMANCE.BANNER_CREATIVE_ID)
                .from(BANNERS_PERFORMANCE)
                .where(BANNERS_PERFORMANCE.BID.in(bannerIds))
                .fetch(BANNERS_PERFORMANCE.BANNER_CREATIVE_ID);
    }

    private List<Long> getCreativeIds(int shard, Collection<Long> bannerIds) {
        return dslContextProvider.ppc(shard)
                .select(BANNERS_PERFORMANCE.CREATIVE_ID)
                .from(BANNERS_PERFORMANCE)
                .where(BANNERS_PERFORMANCE.BID.in(bannerIds))
                .fetch(BANNERS_PERFORMANCE.CREATIVE_ID);
    }

    private List<Long> getMobileContentIds(int shard, Collection<Long> ids) {
        return dslContextProvider.ppc(shard)
                .select(MOBILE_CONTENT.MOBILE_CONTENT_ID)
                .from(MOBILE_CONTENT)
                .where(MOBILE_CONTENT.MOBILE_CONTENT_ID.in(ids))
                .fetch(MOBILE_CONTENT.MOBILE_CONTENT_ID);
    }

    private List<Long> getCalloutIds(int shard, Collection<Long> ids) {
        return dslContextProvider.ppc(shard)
                .select(ADDITIONS_ITEM_CALLOUTS.ADDITIONS_ITEM_ID)
                .from(ADDITIONS_ITEM_CALLOUTS)
                .where(ADDITIONS_ITEM_CALLOUTS.ADDITIONS_ITEM_ID.in(ids))
                .fetch(ADDITIONS_ITEM_CALLOUTS.ADDITIONS_ITEM_ID);
    }

    private List<ModResyncQueueObj> convertIdsToObjects(
            ModResyncQueueObjectType type, List<Long> ids, Long priority, boolean remoderate) {
        return mapList(ids, id -> convertIdToObject(type, id, priority, remoderate));
    }

    private ModResyncQueueObj convertIdToObject(ModResyncQueueObjectType type, Long id,
                                                Long priority, boolean remoderate) {
        return new ModResyncQueueObj()
                .withObjectType(type)
                .withObjectId(id)
                .withPriority(priority)
                .withRemoderate(remoderate);
    }
}
