package ru.yandex.direct.intapi.entity.moderation.service.block;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.model.UsersBlockReasonType;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.intapi.entity.moderation.model.BlockClientResult;
import ru.yandex.direct.intapi.entity.moderation.model.ClientBlockStatus;
import ru.yandex.direct.intapi.entity.moderation.model.UnblockClientResult;
import ru.yandex.direct.intapi.entity.moderation.service.IntapiModerationService;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.PathNode;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.MoreObjects.firstNonNull;
import static java.util.Collections.emptyList;
import static ru.yandex.direct.core.entity.user.utils.BlockedUserUtil.logUserBlock;
import static ru.yandex.direct.core.entity.user.utils.BlockedUserUtil.logUserUnblock;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;

@Service
public class ModerationBlockService {
    private static final Logger logger = LoggerFactory.getLogger(ModerationBlockService.class);

    private final IntapiModerationService intapiModerationService;
    private final UserService userService;
    private static final User OPERATOR = new User().withUid(1L);

    @Autowired
    public ModerationBlockService(IntapiModerationService intapiModerationService,
                                  UserService userService) {
        this.intapiModerationService = intapiModerationService;
        this.userService = userService;
    }

    public Map<ClientId, BlockClientResult> blockClients(List<ClientId> clientIds,
                                                         UsersBlockReasonType blockReasonType,
                                                         String blockComment) {
        BlockHandler blockHandler = new BlockHandler(
                userService.getCampaignIdsForStopping(clientIds),
                blockComment,
                firstNonNull(blockReasonType, UsersBlockReasonType.NOT_SET));

        changeStatusBlocked(false, clientIds, blockHandler);
        return blockHandler.getResults();
    }

    public Map<ClientId, UnblockClientResult> unblockClients(List<ClientId> clientIds) {
        UnblockHandler unblockHandler = new UnblockHandler();
        changeStatusBlocked(true, clientIds, unblockHandler);
        return unblockHandler.getResults();
    }

    private interface IBlockHandler {
        ValidationResult<List<User>, Defect> validateUsers(List<User> users);
        ValidationResult<List<User>, Defect> apply(List<User> users);
        void addResult(ClientId clientId, ClientBlockStatus blockStatus, String error);
        void addNotFoundResult(ClientId clientId);
        void logUser(User user);
    }

    private class BlockHandler implements IBlockHandler {
        private Map<ClientId, BlockClientResult> results;
        private Map<Long, List<Long>> campaignIdsByClientId;
        private String blockComment;
        private UsersBlockReasonType blockReasonType;

        BlockHandler(Map<Long, List<Long>> campaignIdsByClientId, String blockComment,
                     UsersBlockReasonType blockReasonType) {
            this.results = new HashMap<>();
            this.campaignIdsByClientId = campaignIdsByClientId;
            this.blockComment = blockComment;
            this.blockReasonType = blockReasonType;
        }

        public Map<ClientId, BlockClientResult> getResults() {
            return results;
        }

        @Override
        public ValidationResult<List<User>, Defect> validateUsers(List<User> users) {
            return userService.validateUsersToBlock(users, campaignIdsByClientId, true);
        }

        @Override
        public ValidationResult<List<User>, Defect> apply(List<User> users) {
            return userService.blockUsers(OPERATOR, users, campaignIdsByClientId, blockReasonType, blockComment, true);
        }

        @Override
        public void addResult(ClientId clientId, ClientBlockStatus blockStatus, String error) {
            results.put(clientId, new BlockClientResult(clientId.asLong(), blockStatus, error));
        }

        @Override
        public void addNotFoundResult(ClientId clientId) {
            results.put(clientId, new BlockClientResult(clientId.asLong(), "Client not found"));
        }

        @Override
        public void logUser(User user) {
            Long clientId = user.getClientId().asLong();
            logUserBlock(user, OPERATOR, campaignIdsByClientId.getOrDefault(clientId, emptyList()));
        }
    }

    private class UnblockHandler implements IBlockHandler {
        private Map<ClientId, UnblockClientResult> results;

        UnblockHandler() {
            this.results = new HashMap<>();
        }

        public Map<ClientId, UnblockClientResult> getResults() {
            return results;
        }
        @Override
        public ValidationResult<List<User>, Defect> validateUsers(List<User> users) {
            return userService.validateUsersToUnblock(users);
        }

        @Override
        public ValidationResult<List<User>, Defect> apply(List<User> users) {
            return userService.unblockUsers(users);
        }

        @Override
        public void addResult(ClientId clientId, ClientBlockStatus blockStatus, String error) {
            results.put(clientId, new UnblockClientResult(clientId.asLong(), blockStatus, error));
        }

        @Override
        public void addNotFoundResult(ClientId clientId) {
            results.put(clientId, new UnblockClientResult(clientId.asLong(), "Client not found"));
        }

        @Override
        public void logUser(User user) {
            logUserUnblock(user, OPERATOR);
        }
    }

    private void changeStatusBlocked(
            boolean oldStatusBlocked,
            List<ClientId> clientIds,
            IBlockHandler blockHandler
    ) {
        List<User> users = intapiModerationService.getUsersByClientIds(clientIds);
        Map<ClientId, Boolean> previousClientStatusBlock = intapiModerationService.getCurrentStatusBlock(users);
        List<User> filteredUsers = filterList(users, u -> u.getStatusBlocked() == oldStatusBlocked);
        ValidationResult<List<User>, Defect> vr = blockHandler.validateUsers(filteredUsers);
        Map<ClientId, List<String>> clientIdErrors = new HashMap<>();
        List<User> validUsers = new ArrayList<>();

        var subResults = vr.getSubResults();
        for (PathNode pathNode : subResults.keySet()) {
            var validationResult = subResults.get(pathNode);
            if (validationResult.hasAnyErrors()) {
                List<Defect> defects = validationResult.getErrors();
                User user = (User) validationResult.getValue();
                ClientId clientId = user.getClientId();
                clientIdErrors.putIfAbsent(clientId, new ArrayList<>());
                clientIdErrors.get(clientId).add(String.format("%s (%s)", user.getLogin(), defects));
            } else {
                validUsers.add((User) validationResult.getValue());
            }
        }

        if (!validUsers.isEmpty()) {
            ValidationResult<List<User>, Defect> applyResult = blockHandler.apply(validUsers);
            // Здесь уже не должно быть ошибок валидации
            if (applyResult.hasAnyErrors()) {
                List<String> failedLogins = applyResult.getSubResults().values().stream()
                        .filter(ValidationResult::hasAnyErrors)
                        .map(ValidationResult::getValue)
                        .map(o -> (User) o)
                        .map(User::getLogin)
                        .collect(Collectors.toList());
                logger.error("Can't change users block status: validation failed for logins {}", failedLogins);
                throw new RuntimeException("Can't change users block status: validation failed");
            }
        }

        for (ClientId clientId : clientIds) {
            if (previousClientStatusBlock.containsKey(clientId)) {
                List<String> errors = clientIdErrors.get(clientId);
                String errorMessage;
                if (errors != null) {
                    errorMessage = String.format("Errors found for some users: %s", String.join(",", errors));
                } else {
                    errorMessage = null;
                }
                blockHandler.addResult(
                        clientId,
                        previousClientStatusBlock.get(clientId) ? ClientBlockStatus.BLOCKED : ClientBlockStatus.ACTIVE,
                        errorMessage);
            } else {
                blockHandler.addNotFoundResult(clientId);
            }
        }
        validUsers.forEach(blockHandler::logUser);
    }
}
