package ru.yandex.direct.intapi.entity.moderation.service.client;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.dbutil.sharding.ShardedData;
import ru.yandex.direct.intapi.entity.moderation.model.client.ModerationClientParams;
import ru.yandex.direct.intapi.entity.moderation.model.client.ModerationClientsRequest;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.intapi.entity.moderation.model.client.ModerationClientsRequest.CLIENT_IDS;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.maxListSize;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.notEmptyCollection;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.unique;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;

@Service
public class ModerationClientService {

    private static final int MAX_CLIENT_IDS = 1000;

    private final ClientService clientService;
    private final ShardHelper shardHelper;

    @Autowired
    public ModerationClientService(ClientService clientService, ShardHelper shardHelper) {
        this.clientService = clientService;
        this.shardHelper = shardHelper;
    }

    public ValidationResult<ModerationClientsRequest, Defect> validateClientsRequest(ModerationClientsRequest request) {
        ItemValidationBuilder<ModerationClientsRequest, Defect> builder = ItemValidationBuilder.of(request);

        builder.list(request.getClientIds(), CLIENT_IDS)
                .check(notNull())
                .check(notEmptyCollection())
                .check(maxListSize(MAX_CLIENT_IDS));

        if (!builder.getResult().hasAnyErrors()) {
            builder.list(request.getClientIds(), CLIENT_IDS)
                    .checkEach(notNull())
                    .checkEach(validId())
                    .checkEach(unique());
        }

        return builder.getResult();
    }

    public Map<Long, ModerationClientParams> getClientParams(Collection<Long> clientIds) {
        List<Client> coreClients = getCoreClients(clientIds);
        return StreamEx.of(coreClients)
                .mapToEntry(Client::getClientId, this::convertClientToExternalFormat)
                .toMap();
    }

    private List<Client> getCoreClients(Collection<Long> clientIds) {
        ShardedData<Long> shardedClientIds = shardHelper.groupByShard(clientIds, ShardKey.CLIENT_ID);
        List<Client> clients = new ArrayList<>();
        shardedClientIds.forEach((shard, clientIdsOfShardAsLong) -> {
            List<ClientId> clientIdsOfShard = mapList(clientIdsOfShardAsLong, ClientId::fromLong);
            clients.addAll(clientService.getClients(shard, clientIdsOfShard));
        });
        return clients;
    }

    private ModerationClientParams convertClientToExternalFormat(Client client) {
        ModerationClientParams result = new ModerationClientParams();
        result.setTextAutocorrectionEnabled(!client.getNoTextAutocorrection());
        return result;
    }
}
