package ru.yandex.direct.intapi.entity.moderation.service.modedit;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;

import org.jooq.Configuration;
import org.jooq.Field;
import org.jooq.Record;
import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.dbschema.ppc.enums.BannersStatusbssynced;
import ru.yandex.direct.dbschema.ppc.enums.BannersStatusmoderate;
import ru.yandex.direct.dbutil.QueryWithForbiddenShardMapping;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.dbutil.sharding.ShardedData;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.intapi.entity.moderation.model.modedit.ModerationEditFieldStatus;
import ru.yandex.direct.intapi.entity.moderation.model.modedit.ModerationEditObjectResult;
import ru.yandex.direct.intapi.entity.moderation.model.modedit.ModerationEditReplacement;
import ru.yandex.direct.intapi.entity.moderation.model.modedit.ModerationEditStatus;
import ru.yandex.direct.intapi.entity.moderation.model.modedit.ModerationEditType;

import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS;
import static ru.yandex.direct.jooqmapper.JooqMapperUtils.makeCaseStatement;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;

@Component
public class BannerModerationEditHandler implements ModerationEditHandler {

    public static final String TITLE = "title";
    public static final String TITLE_EXTENSION = "title_extension";
    public static final String BODY = "body";

    private static final Set<BannersStatusmoderate> APPLICABLE_STATUSES = Set.of(
            BannersStatusmoderate.Sent,
            BannersStatusmoderate.Yes,
            BannersStatusmoderate.No
    );

    private final DslContextProvider dslContextProvider;
    private final ShardHelper shardHelper;

    @Autowired
    public BannerModerationEditHandler(DslContextProvider dslContextProvider, ShardHelper shardHelper) {
        this.dslContextProvider = dslContextProvider;
        this.shardHelper = shardHelper;
    }

    @Override
    public ModerationEditType getType() {
        return ModerationEditType.BANNER;
    }

    @Override
    @QueryWithForbiddenShardMapping("Есть только bid; нагрузка на ручку невысокая;" +
            "ручка временная до съезда со старого транспорта в Модерацию")
    public List<ModerationEditObjectResult> edit(List<ModerationEditReplacement> replacements) {
        ShardedData<ModerationEditReplacement> shardedData =
                shardHelper.groupByShard(replacements, ShardKey.BID, ModerationEditReplacement::getId);
        return shardedData
                .stream()
                .map(e -> editInShardWithTransaction(e.getKey(), e.getValue()))
                .toFlatList(Function.identity());
    }

    private Collection<ModerationEditObjectResult> editInShardWithTransaction(
            int shard,
            List<ModerationEditReplacement> replacements) {
        List<ModerationEditObjectResult> results = new ArrayList<>();
        dslContextProvider.ppcTransaction(shard, configuration ->
                results.addAll(editInShard(configuration, replacements)));
        return results;
    }

    private Collection<ModerationEditObjectResult> editInShard(Configuration configuration,
                                                               List<ModerationEditReplacement> replacements) {
        Map<Long, ModerationEditReplacement> replacementMap = listToMap(replacements, ModerationEditReplacement::getId);

        Map<Long, BannerWithTexts> actualTexts = DSL.using(configuration)
                .select(BANNERS.BID, BANNERS.TITLE, BANNERS.TITLE_EXTENSION, BANNERS.BODY, BANNERS.STATUS_MODERATE)
                .from(BANNERS)
                .where(BANNERS.BID.in(replacementMap.keySet()))
                .forUpdate()
                .fetchMap(BANNERS.BID, this::mapRecordToObject);

        var results = new HashMap<Long, ModerationEditObjectResult>();

        Map<Long, String> applicableTitles = getApplicableTitles(replacementMap, actualTexts, results);
        Map<Long, String> applicableTitleEx = getApplicableTitleExtensions(replacementMap, actualTexts, results);
        Map<Long, String> applicableBodies = getApplicableBodies(replacementMap, actualTexts, results);

        Set<Long> applicableIds = new HashSet<>(applicableTitles.keySet());
        applicableIds.addAll(applicableTitleEx.keySet());
        applicableIds.addAll(applicableBodies.keySet());

        Field<String> titleUpdateField = makeCaseStatement(BANNERS.BID, BANNERS.TITLE, applicableTitles);
        Field<String> titleExUpdateField = makeCaseStatement(BANNERS.BID, BANNERS.TITLE_EXTENSION, applicableTitleEx);
        Field<String> bodyUpdateField = makeCaseStatement(BANNERS.BID, BANNERS.BODY, applicableBodies);

        DSL.using(configuration)
                .update(BANNERS)
                .set(BANNERS.TITLE, titleUpdateField)
                .set(BANNERS.TITLE_EXTENSION, titleExUpdateField)
                .set(BANNERS.BODY, bodyUpdateField)
                .set(BANNERS.STATUS_MODERATE, BannersStatusmoderate.Ready)
                .set(BANNERS.STATUS_BS_SYNCED, BannersStatusbssynced.No)
                .where(BANNERS.BID.in(applicableIds))
                .execute();

        return results.values();
    }

    private Map<Long, String> getApplicableTitles(Map<Long, ModerationEditReplacement> replacementMap,
                                                  Map<Long, BannerWithTexts> actualTexts,
                                                  Map<Long, ModerationEditObjectResult> results) {
        return getApplicableValues(replacementMap, actualTexts, TITLE, bwt -> bwt.title, results);
    }

    private Map<Long, String> getApplicableTitleExtensions(Map<Long, ModerationEditReplacement> replacementMap,
                                                           Map<Long, BannerWithTexts> actualTexts,
                                                           Map<Long, ModerationEditObjectResult> results) {
        return getApplicableValues(replacementMap, actualTexts, TITLE_EXTENSION, bwt -> bwt.titleExtension, results);
    }

    private Map<Long, String> getApplicableBodies(Map<Long, ModerationEditReplacement> replacementMap,
                                                  Map<Long, BannerWithTexts> actualTexts,
                                                  Map<Long, ModerationEditObjectResult> results) {
        return getApplicableValues(replacementMap, actualTexts, BODY, bwt -> bwt.body, results);
    }

    private Map<Long, String> getApplicableValues(Map<Long, ModerationEditReplacement> replacementMap,
                                                  Map<Long, BannerWithTexts> existingTexts,
                                                  String fieldName,
                                                  Function<BannerWithTexts, String> existingValueExtractor,
                                                  Map<Long, ModerationEditObjectResult> results) {
        var applicableValues = new HashMap<Long, String>();
        replacementMap.forEach((id, replacement) -> {
            var existing = existingTexts.get(id);
            if (!replacement.getOldData().containsKey(fieldName)) {
                return;
            }
            if (existing == null) {
                results.putIfAbsent(id, ModerationEditObjectResult.notFound(id, getType()));
                return;
            }
            if (!APPLICABLE_STATUSES.contains(existing.statusmoderate)) {
                results.putIfAbsent(id, ModerationEditObjectResult.badStatusModerate(id, getType()));
                return;
            }
            String existingText = existingValueExtractor.apply(existing);

            results.putIfAbsent(id, new ModerationEditObjectResult(id, getType()));
            ModerationEditObjectResult result = results.get(id);

            if (Objects.equals(existingText, replacement.getOldData().get(fieldName))) {
                applicableValues.put(id, replacement.getNewData().get(fieldName));
                result.addFieldResult(fieldName, ModerationEditFieldStatus.CHANGED);
                result.setEditStatus(ModerationEditStatus.CHANGED);
            } else {
                if (null == result.getFieldResults()) {
                    result.setEditStatus(ModerationEditStatus.NOT_CHANGED);
                }
                result.addFieldResult(fieldName, ModerationEditFieldStatus.OLD_VALUE_MISMATCH);
            }
        });
        return applicableValues;
    }

    private BannerWithTexts mapRecordToObject(Record rec) {
        return new BannerWithTexts(
                rec.getValue(BANNERS.TITLE),
                rec.getValue(BANNERS.TITLE_EXTENSION),
                rec.getValue(BANNERS.BODY),
                rec.getValue(BANNERS.STATUS_MODERATE));
    }

    private static class BannerWithTexts {
        private final String title;
        private final String titleExtension;
        private final String body;
        private final BannersStatusmoderate statusmoderate;

        BannerWithTexts(String title, String titleExtension, String body, BannersStatusmoderate statusmoderate) {
            this.title = title;
            this.titleExtension = titleExtension;
            this.body = body;
            this.statusmoderate = statusmoderate;
        }
    }
}
