package ru.yandex.direct.intapi.entity.moderation.service.modedit;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.common.util.collections.Pair;
import ru.yandex.direct.intapi.entity.moderation.model.modedit.ModerationEditObjectResult;
import ru.yandex.direct.intapi.entity.moderation.model.modedit.ModerationEditObjectsResult;
import ru.yandex.direct.intapi.entity.moderation.model.modedit.ModerationEditReplacement;
import ru.yandex.direct.intapi.entity.moderation.model.modedit.ModerationEditType;
import ru.yandex.direct.intapi.validation.kernel.ValidationResultConversionService;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.function.Function.identity;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.validation.result.PathHelper.index;

@Service
@SuppressWarnings({"rawtypes"})
public class ModerationEditService {

    private final ModerationEditValidationService validationService;
    private final Map<ModerationEditType, ModerationEditHandler> handlers;
    private final ValidationResultConversionService conversionService;

    @Autowired
    public ModerationEditService(ModerationEditValidationService validationService,
                                 List<ModerationEditHandler> handlers,
                                 ValidationResultConversionService conversionService) {
        this.validationService = validationService;
        this.handlers = listToMap(handlers, ModerationEditHandler::getType);
        this.conversionService = conversionService;
    }

    public ModerationEditObjectsResult editObjects(List<ModerationEditReplacement> replacements) {
        ValidationResult<List<ModerationEditReplacement>, Defect> listVr = validationService.validateList(replacements);
        if (listVr.hasAnyErrors()) {
            return new ModerationEditObjectsResult(conversionService.buildIntapiValidationResult(listVr));
        }

        ValidationResult<List<ModerationEditReplacement>, Defect> itemsVr =
                validationService.validateItems(replacements);

        var subResults = itemsVr.getSubResults();
        var validReplacements = new ArrayList<ModerationEditReplacement>();
        for (int i = 0; i < replacements.size(); ++i) {
            var subResult = subResults.get(index(i));
            if (!subResult.hasAnyErrors()) {
                validReplacements.add(replacements.get(i));
            }
        }

        var resultsByTypeAndId =
                listToMap(execute(validReplacements), this::getKey, identity());

        var allResults = new ArrayList<ModerationEditObjectResult>();

        for (int i = 0; i < replacements.size(); ++i) {
            var itemVr = subResults.get(index(i));
            var replacement = replacements.get(i);
            if (!itemVr.hasAnyErrors()) {
                var result = resultsByTypeAndId.get(getKey(replacement));
                allResults.add(result != null
                        ? result
                        : ModerationEditObjectResult.notFound(replacement.getId(), replacement.getType()));
            } else {
                var convertedVr = conversionService.buildIntapiValidationResult(itemVr);
                allResults.add(new ModerationEditObjectResult(
                        replacement != null ? replacement.getId() : null,
                        replacement != null ? replacement.getType() : null,
                        convertedVr));
            }
        }
        return new ModerationEditObjectsResult(allResults);
    }

    private List<ModerationEditObjectResult> execute(List<ModerationEditReplacement> replacements) {
        var typeToReplacements = StreamEx.of(replacements)
                .mapToEntry(ModerationEditReplacement::getType, identity())
                .grouping();

        List<ModerationEditObjectResult> results = new ArrayList<>();
        typeToReplacements.forEach((type, replacementsOfType) ->
                results.addAll(handlers.get(type).edit(replacementsOfType)));
        return results;
    }

    private Pair<ModerationEditType, Long> getKey(ModerationEditObjectResult moderationEditObjectResult) {
        return new Pair<>(moderationEditObjectResult.getType(), moderationEditObjectResult.getId());
    }

    private Pair<ModerationEditType, Long> getKey(ModerationEditReplacement moderationEditReplacement) {
        return new Pair<>(moderationEditReplacement.getType(), moderationEditReplacement.getId());
    }
}
