package ru.yandex.direct.intapi.entity.moderation.service.modedit;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Predicate;

import one.util.streamex.StreamEx;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.banner.service.validation.BannerTextValidator;
import ru.yandex.direct.intapi.entity.moderation.model.modedit.ModerationEditReplacement;
import ru.yandex.direct.intapi.validation.kernel.TranslatableIntapiDefect;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.presentation.DefaultDefectPresentationRegistry;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.DefectId;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.entity.banner.service.validation.BannerTextConstants.MAX_NUMBER_OF_NARROW_CHARACTERS;
import static ru.yandex.direct.core.entity.banner.type.body.BannerWithBodyConstants.MAX_LENGTH_BODY;
import static ru.yandex.direct.core.entity.banner.type.body.BannerWithBodyConstants.MAX_LENGTH_BODY_WORD;
import static ru.yandex.direct.core.entity.banner.type.title.BannerConstantsService.MAX_LENGTH_TITLE_WORD;
import static ru.yandex.direct.core.entity.banner.type.title.BannerConstantsService.NEW_MAX_LENGTH_TITLE;
import static ru.yandex.direct.core.entity.banner.type.titleextension.BannerWithTitleExtensionConstants.MAX_LENGTH_TITLE_EXTENSION;
import static ru.yandex.direct.intapi.entity.moderation.service.modedit.BannerModerationEditHandler.BODY;
import static ru.yandex.direct.intapi.entity.moderation.service.modedit.BannerModerationEditHandler.TITLE;
import static ru.yandex.direct.intapi.entity.moderation.service.modedit.BannerModerationEditHandler.TITLE_EXTENSION;
import static ru.yandex.direct.intapi.validation.kernel.IntapiDefectPresentationProviders.defaultIntapiDefect;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.collectionSize;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.notEmptyMap;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.unique;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;

@Service
@SuppressWarnings({"rawtypes", "unchecked"})
public class ModerationEditValidationService {

    private static final int IDS_LIMIT = 1000;

    private final BannerTextValidator titleValidator;
    private final BannerTextValidator titleExtensionValidator;
    private final BannerTextValidator bodyValidator;

    public ModerationEditValidationService() {
        titleValidator = BannerTextValidator.builder(NEW_MAX_LENGTH_TITLE, MAX_LENGTH_TITLE_WORD).build();
        titleExtensionValidator = BannerTextValidator.builder(MAX_LENGTH_TITLE_EXTENSION, MAX_LENGTH_TITLE_WORD)
                .withMaxNumberOfNarrowCharacters(MAX_NUMBER_OF_NARROW_CHARACTERS)
                .build();
        bodyValidator = BannerTextValidator.builder(MAX_LENGTH_BODY, MAX_LENGTH_BODY_WORD)
                .withMaxNumberOfNarrowCharacters(MAX_NUMBER_OF_NARROW_CHARACTERS)
                .build();
    }

    public ValidationResult<List<ModerationEditReplacement>, Defect> validateList(
            List<ModerationEditReplacement> replacements) {
        ListValidationBuilder<ModerationEditReplacement, Defect> vb = ListValidationBuilder.of(replacements);
        vb
                .check(notNull())
                .check(collectionSize(1, IDS_LIMIT));
        return vb.getResult();
    }

    public ValidationResult<List<ModerationEditReplacement>, Defect> validateItems(
            List<ModerationEditReplacement> replacements) {
        ListValidationBuilder<ModerationEditReplacement, Defect> vb = ListValidationBuilder.of(replacements);
        vb
                .checkEach(notNull())
                .checkEach(unique(ModerationEditReplacement::getId))
                .checkEachBy(this::validateReplacement, When.isValid());
        return vb.getResult();
    }

    private ValidationResult<ModerationEditReplacement, Defect> validateReplacement(
            ModerationEditReplacement replacement) {
        ItemValidationBuilder<ModerationEditReplacement, Defect> vb = ItemValidationBuilder.of(replacement);

        vb.item(replacement.getId(), ModerationEditReplacement.ID)
                .check(notNull())
                .check(validId());

        vb.item(replacement.getType(), ModerationEditReplacement.TYPE)
                .check(notNull());

        vb.item(replacement.getOldData(), ModerationEditReplacement.OLD_DATA)
                .check(notNull())
                .check(notEmptyMap())
                .check(noNullValuesInData());

        vb.item(replacement.getNewData(), ModerationEditReplacement.NEW_DATA)
                .check(notNull())
                .check(notEmptyMap())
                .check(noNullValuesInData())
                .checkBy(validateBannerNewData(), When.isValid());

        if (!vb.getResult().hasAnyErrors()) {
            vb.check(bothOldAndNewDataContainsSameFields());
        }

        return vb.getResult();
    }

    // сделано максимально просто и хрупко,
    // так как это временный вариант до переезда функциональности в новый транспорт в ~Q4-2021
    private Validator<Map<String, String>, Defect> validateBannerNewData() {
        return newData -> {
            ItemValidationBuilder<Map<String, String>, Defect> vb = ItemValidationBuilder.of(newData);

            vb.item(newData.get(TITLE), TITLE)
                    .checkBy(titleValidator, When.notNull());

            vb.item(newData.get(TITLE_EXTENSION), TITLE_EXTENSION)
                    .checkBy(titleExtensionValidator, When.notNull());

            vb.item(newData.get(BODY), BODY)
                    .checkBy(bodyValidator, When.notNull());

            return vb.getResult();
        };
    }

    private Constraint<Map<String, String>, Defect> noNullValuesInData() {
        Predicate<Map<String, String>> predicate = data -> StreamEx.of(data.values()).noneMatch(Objects::isNull);
        return Constraint.fromPredicate(predicate, dataContainsFieldsWithNullValues());
    }

    private Constraint<ModerationEditReplacement, Defect> bothOldAndNewDataContainsSameFields() {
        Predicate<ModerationEditReplacement> predicate =
                replacement -> {
                    Set<String> oldFields = replacement.getOldData().keySet();
                    Set<String> newFields = replacement.getNewData().keySet();
                    return oldFields.equals(newFields);
                };
        return Constraint.fromPredicate(predicate, oldAndNewDataContainsDifferentFields());
    }

    private Defect dataContainsFieldsWithNullValues() {
        return new Defect(ModEditDefectIds.OLD_AND_NEW_DATA_CONTAINS_DIFFERENT_FIELDS);
    }

    private Defect oldAndNewDataContainsDifferentFields() {
        return new Defect(ModEditDefectIds.OLD_AND_NEW_DATA_CONTAINS_DIFFERENT_FIELDS);
    }

    private enum ModEditDefectIds implements DefectId<Void> {
        OLD_AND_NEW_DATA_CONTAINS_DIFFERENT_FIELDS,
        DATA_CONTAINS_FIELD_WITH_NULL_VALUE
    }

    public static DefaultDefectPresentationRegistry<TranslatableIntapiDefect> modEditDefectPresentationRegistry() {
        return DefaultDefectPresentationRegistry.builder()
                .register(ModEditDefectIds.OLD_AND_NEW_DATA_CONTAINS_DIFFERENT_FIELDS, defaultIntapiDefect())
                .register(ModEditDefectIds.DATA_CONTAINS_FIELD_WITH_NULL_VALUE, defaultIntapiDefect())
                .build();
    }
}
