package ru.yandex.direct.intapi.entity.organization;

import java.util.Collections;
import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.ws.rs.POST;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import one.util.streamex.StreamEx;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.altay.model.language.LanguageOuterClass.Language;
import ru.yandex.direct.intapi.entity.organization.model.OrganizationsCheckAccessRequest;
import ru.yandex.direct.intapi.entity.organization.model.OrganizationsCheckAccessResponse;
import ru.yandex.direct.intapi.entity.organization.model.OrganizationsInfoRequest;
import ru.yandex.direct.intapi.entity.organization.model.OrganizationsInfoResponse;
import ru.yandex.direct.organizations.swagger.OrganizationInfo;
import ru.yandex.direct.organizations.swagger.OrganizationsClient;
import ru.yandex.direct.organizations.swagger.OrganizationsClientException;
import ru.yandex.direct.organizations.swagger.model.PubApiCompany;
import ru.yandex.direct.tvm.AllowServices;
import ru.yandex.direct.web.core.model.WebErrorResponse;
import ru.yandex.direct.web.core.model.WebResponse;

import static org.apache.commons.lang3.StringUtils.isEmpty;
import static ru.yandex.altay.model.language.LanguageOuterClass.Language.RU;
import static ru.yandex.direct.organizations.swagger.OrganizationsClient.getLanguageByName;
import static ru.yandex.direct.tvm.TvmService.DIRECT_AUTOTESTS;
import static ru.yandex.direct.tvm.TvmService.DIRECT_DEVELOPER;
import static ru.yandex.direct.tvm.TvmService.DIRECT_SCRIPTS_PROD;
import static ru.yandex.direct.tvm.TvmService.DIRECT_SCRIPTS_TEST;
import static ru.yandex.direct.tvm.TvmService.DIRECT_WEB_PROD;
import static ru.yandex.direct.tvm.TvmService.DIRECT_WEB_TEST;

@RestController
@RequestMapping(
        name = "organizations",
        path = "/organizations",
        produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
@Api(value = "API для получения данных по организациям справочника")
@ParametersAreNonnullByDefault
public class OrganizationsController {

    private final OrganizationsClient organizationsClient;

    public OrganizationsController(OrganizationsClient organizationsClient) {
        this.organizationsClient = organizationsClient;
    }

    @POST
    @ApiOperation(
            value = "Возвращает информацию об организациях Справочника по их пермалинкам",
            nickname = "get_info",
            httpMethod = "POST"
    )
    @ApiResponses({
            @ApiResponse(code = 200, message = "список информации по организациям",
                    response = OrganizationsInfoResponse.class),
            @ApiResponse(code = 400, message = "Bad params", response = WebErrorResponse.class)})
    @RequestMapping(path = "get_info", method = RequestMethod.POST)
    @AllowServices(production = {DIRECT_WEB_PROD, DIRECT_SCRIPTS_PROD},
            testing = {DIRECT_DEVELOPER, DIRECT_AUTOTESTS, DIRECT_WEB_TEST, DIRECT_SCRIPTS_TEST})
    public WebResponse getInfo(@RequestBody OrganizationsInfoRequest request) {
        if (request.getPermalinkIds().isEmpty() || isEmpty(request.getLanguageName())) {
            return new WebErrorResponse(400, "Invalid parameters");
        }

        Language language = getLanguageByName(request.getLanguageName()).orElse(RU);
        try {
            List<OrganizationInfo> result = organizationsClient
                    .getOrganizationsInfo(request.getPermalinkIds(), language, null);
            return new OrganizationsInfoResponse(result);
        } catch (OrganizationsClientException e) {
            return new OrganizationsInfoResponse(Collections.emptyList());
        }
    }

    @Deprecated //Удалить после того как уберем из перла проверку прав.
    @POST
    @ApiOperation(
            value = "Возвращает информацию о доступности организаций Справочника для пользователей",
            nickname = "check_access",
            httpMethod = "POST"
    )
    @ApiResponses({
            @ApiResponse(code = 200, message = "список информации о доступности организаций для пользователей",
                    response = OrganizationsInfoResponse.class),
            @ApiResponse(code = 400, message = "Bad params", response = WebErrorResponse.class)})
    @RequestMapping(path = "check_access", method = RequestMethod.POST)
    @AllowServices(production = {DIRECT_WEB_PROD}, testing = {DIRECT_DEVELOPER, DIRECT_AUTOTESTS, DIRECT_WEB_TEST})
    public WebResponse checkAccess(@RequestBody OrganizationsCheckAccessRequest request) {
        if (request.getPermalinkIds().isEmpty() || request.getUids().isEmpty()) {
            return new WebErrorResponse(400, "Invalid parameters");
        }
        List<Long> permalinkIds;
        try {
            permalinkIds = StreamEx.of(organizationsClient
                    .getMultipleOrganizationsInfo(request.getPermalinkIds(), Collections.emptyList(),
                            Language.RU.name().toLowerCase(), null)
                    .getCompanies())
                    .map(PubApiCompany::getId)
                    .toList();
        } catch (OrganizationsClientException e) {
            permalinkIds = Collections.emptyList();
        }
        final List<Long> result = permalinkIds;
        Map<Long, List<Long>> permalinkIdsByUid = StreamEx.of(request.getUids())
                .toMap(
                        uid -> uid,
                        uid -> result
                );

        return new OrganizationsCheckAccessResponse(permalinkIdsByUid);
    }
}
