package ru.yandex.direct.intapi.entity.payment.controller;

import javax.ws.rs.POST;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.direct.dbutil.model.LoginOrUid;
import ru.yandex.direct.intapi.entity.payment.model.GetPaymentFormUrlOrInvoiceIdRequest;
import ru.yandex.direct.intapi.entity.payment.model.GetPaymentFormUrlOrInvoiceIdResponse;
import ru.yandex.direct.intapi.entity.payment.model.GetPaymentMethodsRequest;
import ru.yandex.direct.intapi.entity.payment.model.GetPaymentMethodsResponse;
import ru.yandex.direct.intapi.entity.payment.service.IntapiPaymentService;
import ru.yandex.direct.tvm.AllowServices;
import ru.yandex.direct.web.core.model.WebErrorResponse;
import ru.yandex.direct.web.core.model.WebResponse;

import static ru.yandex.direct.tvm.TvmService.DIRECT_AUTOTESTS;
import static ru.yandex.direct.tvm.TvmService.DIRECT_DEVELOPER;
import static ru.yandex.direct.tvm.TvmService.SMB;

@RestController
@RequestMapping(
        name = "payment",
        path = "/payment",
        produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
@Api(value = "API для выставления счета")
public class PaymentController {
    private static final Logger logger = LoggerFactory.getLogger(PaymentController.class);

    private IntapiPaymentService paymentService;

    @Autowired
    public PaymentController(IntapiPaymentService paymentService) {
        this.paymentService = paymentService;
    }

    @POST
    @ApiOperation(
            value = "Получает доступные пользователю способы оплаты счета.",
            nickname = "getPaymentMethods",
            httpMethod = "POST"
    )
    @ApiResponses({
            @ApiResponse(code = 200, message = "доступные способы оплаты счета", response =
                    GetPaymentMethodsResponse.class)})
    @RequestMapping(path = "getPaymentMethods", method = RequestMethod.POST)
    @AllowServices(production = {SMB}, testing = {DIRECT_DEVELOPER, DIRECT_AUTOTESTS, SMB})
    public WebResponse getPaymentMethods(
            @ApiParam(value = "запрос на получение доступных способов оплаты счета", required = true)
            @RequestBody GetPaymentMethodsRequest request) {
        try {
            return paymentService.getPaymentMethods(LoginOrUid.of(request.getLogin(), request.getUid()),
                    request.getIsLegalPerson());
        } catch (Exception e) {
            logger.error("Exception while getting payment methods", e);
            return new WebErrorResponse(1000, e.getMessage());
        }
    }

    @POST
    @ApiOperation(
            value = "Создает ссылку на страницу оплаты по карте или счет в балансе для остальных типов оплаты " +
                    "(при этом создается плательщик, если его еще нет).",
            nickname = "getPaymentFormUrlOrInvoiceId",
            httpMethod = "POST"
    )
    @ApiResponses({
            @ApiResponse(code = 200, message = "ссылка на оплату или идентификатор счета",
                    response = GetPaymentFormUrlOrInvoiceIdResponse.class)})
    @RequestMapping(path = "getPaymentFormUrlOrInvoiceId", method = RequestMethod.POST)
    @AllowServices(production = {SMB}, testing = {DIRECT_DEVELOPER, DIRECT_AUTOTESTS, SMB})
    public WebResponse getPaymentFormUrlOrInvoiceId(
            @ApiParam(value = "запрос на получение ссылки для оплаты", required = true)
            @RequestBody GetPaymentFormUrlOrInvoiceIdRequest request) {
        try {
            return paymentService.getPaymentFormUrlOrInvoiceId(LoginOrUid.of(request.getLogin(), request.getUid()),
                    request.getIsLegalPerson(), request.getIsInvoicePayment(), request.getPaysysId(),
                    request.getPaymentSum(), request.getPromocode());
        } catch (Exception e) {
            logger.error("Exception while creating payment form", e);
            return new WebErrorResponse(1000, e.getMessage());
        }
    }
}
