package ru.yandex.direct.intapi.entity.payment.service;

import java.math.BigDecimal;

import javax.annotation.Nonnull;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.balance.client.model.response.PaymentMethodDetailsFull;
import ru.yandex.direct.core.entity.campaign.repository.WalletRepository;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.promocodes.service.PromocodeHelper;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.core.service.integration.balance.BalanceService;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.model.LoginOrUid;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.intapi.client.IntApiClient;
import ru.yandex.direct.intapi.entity.payment.model.GetPaymentFormUrlOrInvoiceIdResponse;
import ru.yandex.direct.intapi.entity.payment.model.GetPaymentMethodsResponse;
import ru.yandex.direct.intapi.entity.payment.model.PaymentMethod;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.PassportUtils.normalizeLogin;

@Service
public class IntapiPaymentService {
    private final UserService userService;
    private final ShardHelper shardHelper;
    private final ClientService clientService;
    private final WalletRepository walletRepository;
    private final BalanceService balanceService;
    private final IntApiClient intApiClient;
    private final PromocodeHelper promocodeHelper;

    // Фича на доступность промокодов раскатана 100% и для мобильных, и для десктопов, поэтому значение неважно,
    // а возвращать будем оба типа ссылок для оплаты
    private static final boolean IS_MOBILE = false;

    @Autowired
    public IntapiPaymentService(UserService userService,
                                ShardHelper shardHelper,
                                ClientService clientService,
                                WalletRepository walletRepository,
                                BalanceService balanceService,
                                IntApiClient intApiClient,
                                PromocodeHelper promocodeHelper) {
        this.userService = userService;
        this.shardHelper = shardHelper;
        this.clientService = clientService;
        this.walletRepository = walletRepository;
        this.balanceService = balanceService;
        this.intApiClient = intApiClient;
        this.promocodeHelper = promocodeHelper;
    }

    public GetPaymentMethodsResponse getPaymentMethods(@Nonnull LoginOrUid loginOrUid, Boolean isLegalPerson) {
        User user = loginOrUid.map(login -> userService.getUserByLogin(normalizeLogin(login)), userService::getUser);
        checkNotNull(user, "no user found");
        checkNotNull(isLegalPerson, "no legal entity status");
        var clientId = user.getClientId();
        var operatorUid = user.getUid();

        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        CurrencyCode clientCurrency = clientService.getWorkCurrency(clientId).getCode();
        Long walletCid = walletRepository.getActualClientWalletId(shard, clientId, clientCurrency);
        checkNotNull(walletCid, "no client wallet");

        Long personIdToPay = balanceService.getOrCreatePerson(user, operatorUid, isLegalPerson);
        checkNotNull(personIdToPay, "no person to pay");

        var paymentMethodDetailsFullList = balanceService.getPaymentOptionsFull(operatorUid, clientId, walletCid,
                personIdToPay);
        var paymentMethods = mapList(paymentMethodDetailsFullList, IntapiPaymentService::toPaymentMethod);

        return new GetPaymentMethodsResponse(paymentMethods);
    }

    private static PaymentMethod toPaymentMethod(PaymentMethodDetailsFull pmd) {
        var paymentMethod = pmd.getPaymentMethod();
        return new PaymentMethod()
                .withCode(pmd.getCode())
                .withDisabledReasons(pmd.getDisabledReasons())
                .withId(pmd.getId())
                .withLegalEntity(pmd.getLegalEntity())
                .withName(pmd.getName())
                .withPaymentLimit(pmd.getPaymentLimit())
                .withPaymentMethodCode(paymentMethod.getCode())
                .withPaymentMethodId(paymentMethod.getId())
                .withPaymentMethodName(paymentMethod.getName())
                .withRegionId(pmd.getRegionId())
                .withResident(pmd.getResident());
    }

    public GetPaymentFormUrlOrInvoiceIdResponse getPaymentFormUrlOrInvoiceId(@Nonnull LoginOrUid loginOrUid,
                                                                             Boolean isLegalPerson,
                                                                             Boolean isInvoicePayment, Long paysysId,
                                                                             BigDecimal sum, String promocode) {
        checkNotNull(isInvoicePayment, "no invoice payment status");
        checkNotNull(sum, "sum must not be null");
        checkArgument(sum.compareTo(BigDecimal.ZERO) > 0, "sum is negative");

        User user = loginOrUid.map(login -> userService.getUserByLogin(normalizeLogin(login)), userService::getUser);
        checkNotNull(user, "no user found");
        checkNotNull(isLegalPerson, "no legal entity status");
        var clientId = user.getClientId();
        var operatorUid = user.getUid();

        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        CurrencyCode clientCurrency = clientService.getWorkCurrency(clientId).getCode();
        Long walletCid = walletRepository.getActualClientWalletId(shard, clientId, clientCurrency);
        checkNotNull(walletCid, "no client wallet");

        checkState(intApiClient.validatePayCamp(operatorUid, walletCid, operatorUid, sum), "invalid payment");

        Long personIdToPay = balanceService.getOrCreatePerson(user, operatorUid, isLegalPerson);
        checkNotNull(personIdToPay, "no person to pay");

        var denyPromocode = isDeniedPromocode(clientId, promocode);
        if (!isInvoicePayment) {
            // ссылки на оплату картой через траст
            var createAndPayRequestResult = balanceService.createAndPayRequest(operatorUid, clientId,
                    walletCid, clientCurrency, sum, personIdToPay, null, true, promocode, denyPromocode);
            return new GetPaymentFormUrlOrInvoiceIdResponse()
                    .withPaymentUrl(createAndPayRequestResult.getPaymentUrl(false))
                    .withMobilePaymentUrl(createAndPayRequestResult.getPaymentUrl(true));
        } else {
            checkNotNull(paysysId, "no payment system selected for invoice");
            // id счета в балансе для оплаты
            var createInvoiceRequestResult = balanceService.createInvoice(operatorUid, clientId, walletCid,
                    paysysId, sum, personIdToPay, true, promocode, denyPromocode);
            return new GetPaymentFormUrlOrInvoiceIdResponse()
                    .withInvoiceId(createInvoiceRequestResult.getInvoiceId());
        }
    }

    private boolean isDeniedPromocode(ClientId clientId, String promocode) {
        var promocodeValidationContainer = promocodeHelper.preparePromocodeValidationContainer(clientId);
        return promocode == null
                || !promocodeHelper.isApplicablePromocode(promocode, promocodeValidationContainer)
                || promocodeHelper.doDenyPromocode(promocodeValidationContainer);
    }
}
