package ru.yandex.direct.intapi.entity.placement.controller;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import com.google.common.collect.ImmutableSet;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.direct.common.net.NetworkName;
import ru.yandex.direct.core.entity.placements.model1.Placement;
import ru.yandex.direct.core.entity.placements.model1.PlacementType;
import ru.yandex.direct.core.entity.placements.repository.PlacementRepository;
import ru.yandex.direct.intapi.entity.placement.model.PlacementsResponse;
import ru.yandex.direct.web.core.security.netacl.AllowNetworks;

import static com.google.common.base.Preconditions.checkArgument;

@Api(tags = "placements")
@RestController
@RequestMapping(value = "placements", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
@AllowNetworks(NetworkName.DIRECT_PRODUCTION)
public class PlacementController {

    private static final ImmutableSet AVAILABLE_GET_TYPES = ImmutableSet.of(
            PlacementType.OUTDOOR
    );

    private final PlacementRepository placementRepository;

    @Autowired
    public PlacementController(PlacementRepository placementRepository) {
        this.placementRepository = placementRepository;
    }

    @ApiOperation(value = "Get placements by type or list of ids", httpMethod = "GET")
    @ApiResponses(
            @ApiResponse(code = 200, message = "Ok", response = PlacementsResponse.class)
    )
    @RequestMapping(path = "/", method = RequestMethod.GET)
    public PlacementsResponse getPlacements(
            @RequestParam(value = "type", required = false) PlacementType placementType,
            @RequestParam(value = "ids", required = false) Set<Long> ids) {
        Map<Long, Placement> placements;
        if (placementType != null) {
            checkArgument(AVAILABLE_GET_TYPES.contains(placementType),
                    "get placements of type %s is denied", placementType);
            placements = placementRepository.getPlacementsByType(placementType);
        } else if (ids != null) {
            placements = placementRepository.getPlacements(ids);
        } else {
            placements = new HashMap<>();
        }

        return new PlacementsResponse()
                .withPlacements(new ArrayList<>(placements.values()));
    }
}
