package ru.yandex.direct.intapi.entity.queryrec.controller;

import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.ws.rs.POST;
import javax.ws.rs.core.MediaType;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.queryrec.QueryrecService;
import ru.yandex.direct.tvm.AllowServices;

import static ru.yandex.direct.tvm.TvmService.DIRECT_API_PROD;
import static ru.yandex.direct.tvm.TvmService.DIRECT_API_TEST;
import static ru.yandex.direct.tvm.TvmService.DIRECT_DEVELOPER;
import static ru.yandex.direct.tvm.TvmService.DIRECT_INTAPI_TEST;
import static ru.yandex.direct.tvm.TvmService.DIRECT_SCRIPTS_PROD;
import static ru.yandex.direct.tvm.TvmService.DIRECT_SCRIPTS_TEST;
import static ru.yandex.direct.tvm.TvmService.DIRECT_WEB_PROD;
import static ru.yandex.direct.tvm.TvmService.DIRECT_WEB_TEST;

@RestController
@RequestMapping("queryrec")
@Api(value = "API для определения языка текста")
public class QueryrecController {
    private static final Logger logger = LoggerFactory.getLogger(QueryrecController.class);

    @Autowired
    private QueryrecService queryrecService;

    @Autowired
    private ClientService clientService;

    /**
     * Определение языков, на котором написаны тексты
     *
     * @param texts список текстов
     * @return соответствующий список с мапой из языков и вероятностями, что текст написан на этом языке
     */
    @POST
    @ApiOperation(
            value = "определение языка текста",
            nickname = "queryrec",
            httpMethod = "POST"
    )
    @RequestMapping(path = "recognize_texts",
            method = RequestMethod.POST,
            consumes = MediaType.APPLICATION_JSON,
            produces = MediaType.APPLICATION_JSON
    )
    public List<Map<String, Double>> recognizeTexts(
            @ApiParam(value = "список текстов", required = true)
            @RequestBody @Nonnull List<String> texts) {
        logger.debug("process request recognize_texts()");

        return texts.stream()
                .map(queryrecService::getLangProbabilities)
                .map(probabilities -> EntryStream.of(probabilities)
                        .mapKeys(lang -> lang.getIso639Code().toString())
                        .toMap()
                ).collect(Collectors.toList());
    }

    /**
     * Определяет язык текстов.
     *
     * Отличие от метода recognize_texts в том, что recognize_texts не использует алгоритм определения языка
     * из {@link QueryrecService#recognize(String, ClientId, Long)} и возвращает только вероятности.
     *
     * @param clientIdLong id клиента
     * @param texts        список текстов
     * @return соответствующий список определенных языков
     */
    @POST
    @ApiOperation(
            value = "определение языка текста",
            nickname = "queryrec",
            httpMethod = "POST"
    )
    @RequestMapping(path = "analyze_texts_lang",
            method = RequestMethod.POST,
            consumes = MediaType.APPLICATION_JSON,
            produces = MediaType.APPLICATION_JSON
    )
    @AllowServices(
            production = {DIRECT_WEB_PROD, DIRECT_SCRIPTS_PROD, DIRECT_API_PROD},
            testing = {DIRECT_WEB_TEST, DIRECT_SCRIPTS_TEST, DIRECT_API_TEST, DIRECT_INTAPI_TEST, DIRECT_DEVELOPER}
    )
    public List<String> analyzeTextsLang(
            @ApiParam(value = "id клиента")
            @RequestParam("client_id") @Nullable Long clientIdLong,
            @ApiParam(value = "список текстов", required = true)
            @RequestBody @Nonnull List<String> texts) {
        logger.debug("process request analyze_text_lang()");

        ClientId clientId = ClientId.fromNullableLong(clientIdLong);
        Long clientRegionId = clientId != null ?
                clientService.getCountryRegionIdByClientId(clientId).orElse(null) : null;

        return StreamEx.of(texts)
                .map(text -> queryrecService.recognize(text, clientId, clientRegionId))
                .map(language -> language.getIso639Code().toString())
                .toList();
    }
}
