package ru.yandex.direct.intapi.entity.recommendation.controller.service;

import java.time.Duration;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.spring.TestingComponent;
import ru.yandex.direct.intapi.entity.recommendation.controller.model.Recommendation;
import ru.yandex.direct.web.core.model.WebResponse;
import ru.yandex.direct.web.core.model.WebSuccessResponse;
import ru.yandex.direct.ytcomponents.config.DirectYtDynamicConfig;
import ru.yandex.direct.ytwrapper.YtPathUtil;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.direct.ytwrapper.model.YtDynamicOperator;
import ru.yandex.yt.rpcproxy.ETransactionType;
import ru.yandex.yt.ytclient.proxy.ApiServiceTransactionOptions;
import ru.yandex.yt.ytclient.proxy.ModifyRowsRequest;
import ru.yandex.yt.ytclient.tables.ColumnValueType;
import ru.yandex.yt.ytclient.tables.TableSchema;

import static java.util.Arrays.asList;
import static java.util.Collections.singletonList;
import static org.apache.commons.lang3.ObjectUtils.defaultIfNull;

@Service
@ParametersAreNonnullByDefault
@TestingComponent
public class RecommendationInsertService {
    private static final TableSchema SCHEMA = new TableSchema.Builder()
            .addKey("client_id", ColumnValueType.INT64)
            .addKey("type", ColumnValueType.INT64)
            .addKey("cid", ColumnValueType.INT64)
            .addKey("pid", ColumnValueType.INT64)
            .addKey("bid", ColumnValueType.INT64)
            .addKey("user_key_1", ColumnValueType.STRING)
            .addKey("user_key_2", ColumnValueType.STRING)
            .addKey("user_key_3", ColumnValueType.STRING)
            .addKey("timestamp", ColumnValueType.INT64)
            .addValue("data", ColumnValueType.STRING)
            .build()
            .toWrite();

    private static final Duration CONNECT_TIMEOUT = Duration.ofSeconds(10);

    private YtProvider ytProvider;
    private DirectYtDynamicConfig directYtDynamicConfig;

    @Autowired
    public RecommendationInsertService(YtProvider ytProvider, DirectYtDynamicConfig directYtDynamicConfig) {
        this.ytProvider = ytProvider;
        this.directYtDynamicConfig = directYtDynamicConfig;
    }

    public WebResponse createRecommendation(Recommendation rec) {
        ApiServiceTransactionOptions txOpts = new ApiServiceTransactionOptions(ETransactionType.TT_TABLET)
                .setSticky(true)
                .setPing(true)
                .setTimeout(CONNECT_TIMEOUT);

        String path =
                YtPathUtil.generatePath(directYtDynamicConfig.tables().recommendations().baseDir(), "current");

        Long timestamp = defaultIfNull(rec.getTimestamp(), System.currentTimeMillis() / 1000);
        Iterable<? extends List<?>> rows = singletonList(asList(
                rec.getClientId(),
                rec.getType(),
                rec.getCid(),
                rec.getPid(),
                rec.getBid(),
                rec.getUserKey1(),
                rec.getUserKey2(),
                rec.getUserKey3(),
                timestamp,
                rec.getData()));

        ModifyRowsRequest request = new ModifyRowsRequest(path, SCHEMA)
                .setRequireSyncReplica(false)
                .addInserts(rows);

        for (YtCluster cluster : directYtDynamicConfig.getClusters()) {
            YtDynamicOperator dynamicOperator = ytProvider.getDynamicOperator(cluster);
            dynamicOperator
                    .runInTransaction(tx -> tx.modifyRows(request).join(), txOpts); // IGNORE-BAD-JOIN DIRECT-149116
        }

        return new WebSuccessResponse();
    }
}
