package ru.yandex.direct.intapi.entity.rmp.adgroups;

import java.util.List;
import java.util.Map;
import java.util.Set;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.direct.core.aggregatedstatuses.AggregatedStatusesViewService;
import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.service.AdGroupService;
import ru.yandex.direct.core.entity.aggregatedstatuses.adgroup.AggregatedStatusAdGroupData;
import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.uac.service.UacAdGroupService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.intapi.entity.rmp.RmpControllerHelper;
import ru.yandex.direct.intapi.validation.kernel.ValidationResultConversionService;
import ru.yandex.direct.intapi.validation.model.IntapiValidationResponse;
import ru.yandex.direct.tvm.AllowServices;

import static ru.yandex.direct.tvm.TvmService.DIRECT_DEVELOPER;
import static ru.yandex.direct.tvm.TvmService.RMP_PROD;
import static ru.yandex.direct.tvm.TvmService.RMP_TEST;

@RestController
@Api(value = "Контроллер для обновления настроек групп объявлений для рекламы мобильных приложений")
@RequestMapping(value = "/rmp/v1/adgroups")
@AllowServices(production = {RMP_PROD}, testing = {RMP_TEST, DIRECT_DEVELOPER})
public class RmpAdGroupController {
    private final RmpControllerHelper rmpControllerHelper;
    private final ValidationResultConversionService validationResultConversionService;
    private final AdGroupService adGroupService;
    private final RmpAdGroupConverter rmpAdGroupConverter;
    private final AggregatedStatusesViewService aggregatedStatusesViewService;
    private final ShardHelper shardHelper;
    private final UacAdGroupService uacAdGroupService;

    @Autowired
    public RmpAdGroupController(
            RmpControllerHelper rmpControllerHelper,
            ValidationResultConversionService validationResultConversionService,
            AdGroupService adGroupService,
            RmpAdGroupConverter rmpAdGroupConverter,
            AggregatedStatusesViewService aggregatedStatusesViewService,
            ShardHelper shardHelper,
            UacAdGroupService uacAdGroupService) {
        this.rmpControllerHelper = rmpControllerHelper;
        this.validationResultConversionService = validationResultConversionService;
        this.adGroupService = adGroupService;
        this.rmpAdGroupConverter = rmpAdGroupConverter;
        this.aggregatedStatusesViewService = aggregatedStatusesViewService;
        this.shardHelper = shardHelper;
        this.uacAdGroupService = uacAdGroupService;
    }

    @ApiOperation(
            httpMethod = "GET",
            value = "Получить список групп объявлений для рекламы мобильных приложений",
            notes = "Возвращает список групп объявлений"
    )
    @RequestMapping(path = "",
            method = RequestMethod.GET,
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    public List<RmpAdGroupResponse> getRmpAdGroups(
            @ApiParam(value = "ad_group_ids", required = false) @RequestParam List<Long> adGroupIds,
            @ApiParam(value = "client_id", required = true) @RequestParam Long rawClientId
    ) {
        ClientId clientId = ClientId.fromLong(rawClientId);

        List<AdGroup> adGroups = adGroupService.getAdGroups(clientId, adGroupIds);
        Map<Long, AggregatedStatusAdGroupData> statusDataById = aggregatedStatusesViewService.getAdGroupStatusesByIds(
                shardHelper.getShardByClientId(clientId), Set.copyOf(adGroupIds));

        return rmpAdGroupConverter.toRmpAdGroupResponses(adGroups, statusDataById);
    }

    @ApiOperation(
            httpMethod = "PATCH",
            value = "Изменить настройки группы объявлений",
            notes = "Возвращает результат операции"
    )
    @RequestMapping(path = "",
            method = RequestMethod.PATCH,
            consumes = MediaType.APPLICATION_JSON_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    public IntapiValidationResponse updateRmpCampaign(
            @ApiParam(value = "client_id", required = true) @RequestParam Long rawClientId,
            @ApiParam(value = "uid", required = true) @RequestParam Long uid,
            @ApiParam(value = "ad_group_ids", required = true) @RequestParam List<Long> adGroupIds,
            @RequestBody UpdateRmpAdGroupRequest request
    ) {
        ClientId clientId = ClientId.fromLong(rawClientId);
        Client client = rmpControllerHelper.getClient(clientId);
        User operator = rmpControllerHelper.getUser(clientId, uid);

        var operationResult = uacAdGroupService.updateMobileContentAdGroups(
                client, operator.getUid(), adGroupIds, request.getRegions(), null,
                rmpAdGroupConverter.toKeywords(request.getKeywords()), request.getMinusKeywords(), null, null, null);

        return validationResultConversionService.buildValidationResponse(operationResult);
    }
}
