package ru.yandex.direct.intapi.entity.rmp.banner;

import java.math.BigInteger;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.direct.core.entity.banner.model.Banner;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.service.moderation.BannerModerateService;
import ru.yandex.direct.core.entity.campaign.model.MobileContentCampaign;
import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.core.entity.retargeting.Constants;
import ru.yandex.direct.core.entity.retargeting.model.TargetInterest;
import ru.yandex.direct.core.entity.retargeting.repository.TargetingCategoriesCache;
import ru.yandex.direct.core.entity.uac.service.UacAdGroupService;
import ru.yandex.direct.core.entity.uac.service.UacBannerService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.intapi.IntApiException;
import ru.yandex.direct.intapi.entity.rmp.RmpControllerHelper;
import ru.yandex.direct.intapi.validation.kernel.ValidationResultConversionService;
import ru.yandex.direct.intapi.validation.model.IntapiValidationResponse;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.tvm.AllowServices;
import ru.yandex.direct.web.core.model.WebResponse;

import static com.google.common.base.Preconditions.checkNotNull;
import static ru.yandex.direct.tvm.TvmService.DIRECT_DEVELOPER;
import static ru.yandex.direct.tvm.TvmService.RMP_PROD;
import static ru.yandex.direct.tvm.TvmService.RMP_TEST;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@RestController
@Api(value = "Контроллер для создания объявлений о рекламе мобильных приложениях")
@RequestMapping(value = "/rmp/v1/banners")
@AllowServices(production = {RMP_PROD}, testing = {RMP_TEST, DIRECT_DEVELOPER})
public class RmpBannerController {
    private final RmpBannerConverter rmpBannerConverter;
    private final RmpControllerHelper rmpControllerHelper;
    private final RmpBannerService rmpBannerService;
    private final RmpBannerValidationService rmpBannerValidationService;
    private final BannerModerateService bannerModerateService;
    private final ValidationResultConversionService validationResultConversionService;
    private final UacBannerService uacBannerService;
    private final UacAdGroupService uacAdGroupService;
    private final TargetingCategoriesCache targetingCategoriesCache;

    @Autowired
    @SuppressWarnings("checkstyle:parameternumber")
    public RmpBannerController(
            RmpBannerConverter rmpBannerConverter,
            RmpControllerHelper rmpControllerHelper,
            RmpBannerService rmpBannerService,
            RmpBannerValidationService rmpBannerValidationService,
            BannerModerateService bannerModerateService,
            ValidationResultConversionService validationResultConversionService,
            UacBannerService uacBannerService,
            UacAdGroupService uacAdGroupService,
            TargetingCategoriesCache targetingCategoriesCache) {
        this.rmpBannerConverter = rmpBannerConverter;
        this.rmpControllerHelper = rmpControllerHelper;
        this.rmpBannerService = rmpBannerService;
        this.rmpBannerValidationService = rmpBannerValidationService;
        this.bannerModerateService = bannerModerateService;
        this.validationResultConversionService = validationResultConversionService;
        this.uacBannerService = uacBannerService;
        this.uacAdGroupService = uacAdGroupService;
        this.targetingCategoriesCache = targetingCategoriesCache;
    }

    @ApiOperation(
            httpMethod = "GET",
            value = "Получить список баннеров для рекламы мобильных приложений",
            notes = "Возвращает список баннеров"
    )
    @RequestMapping(path = "",
            method = RequestMethod.GET,
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    public List<RmpBannerResponse> getRmpBanners(
            @ApiParam(value = "ad_group_id", required = false) @RequestParam @Nullable Long adGroupId,
            @ApiParam(value = "banner_ids", required = false) @RequestParam @Nullable List<Long> bannerIds,
            @ApiParam(value = "client_id", required = true) @RequestParam Long rawClientId,
            @ApiParam(value = "uid", required = true) @RequestParam Long uid
    ) {
        ClientId clientId = ClientId.fromLong(rawClientId);
        User operator = rmpControllerHelper.getUser(clientId, uid);
        if (adGroupId == null && bannerIds == null) {
            throw new IntApiException(HttpStatus.BAD_REQUEST, "adGroupId and bannerIds are empty");
        }
        if (adGroupId != null) {
            rmpBannerValidationService.validateAdGroupId(adGroupId);
        }

        List<BannerWithSystemFields> banners = getBanners(operator.getUid(), clientId, adGroupId, bannerIds);
        List<Long> currentBannerIds = StreamEx.of(banners).map(Banner::getId).toList();

        return rmpBannerConverter
                .toRmpBannerResponses(banners, uacBannerService.getModerationDiagsByBannerIds(clientId,
                        currentBannerIds));
    }

    @SuppressWarnings("checkstyle:SimplifyBooleanExpression")
    @ApiOperation(
            httpMethod = "POST",
            value = "Создать баннер для рекламы мобильного приложения",
            notes = "Возвращает документ с информацией о созданном баннере"
    )
    @RequestMapping(path = "",
            method = RequestMethod.POST,
            consumes = MediaType.APPLICATION_JSON_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    public WebResponse addRmpBanner(
            @ApiParam(value = "client_id", required = true) @RequestParam Long rawClientId,
            @ApiParam(value = "uid", required = true) @RequestParam Long uid,
            @RequestBody AddRmpBannerRequest request
    ) {
        ClientId clientId = ClientId.fromLong(rawClientId);
        Client client = rmpControllerHelper.getClient(clientId);
        User operator = rmpControllerHelper.getUser(clientId, uid);
        rmpBannerValidationService.validateRequest(clientId, request);

        MobileContentCampaign campaign = rmpBannerService.getCampaign(clientId, request.getCampaignId());
        BannerWithSystemFields banner = request.getHtml5() == null || !request.getHtml5()
                ? rmpBannerConverter.toMobileAppBanner(request)
                : rmpBannerConverter.toImageBanner(request);
        List<TargetInterest> targetInterests = request.getTargetInterests() != null
                ? request.getTargetInterests().stream()
                    .map(it -> new TargetInterest().withInterestId(
                            targetingCategoriesCache.getCategoryByImportId(BigInteger.valueOf(it)))
                            .withCampaignId(campaign.getId()))
                    .collect(Collectors.toList())
                : null;
        if (request.getAdGroupId() == null) {
            createAdGroup(operator, client, campaign, banner, request, targetInterests);
        } else {
            createBanner(operator, clientId, campaign, banner, request.getAdGroupId());
        }
        checkNotNull(banner.getId(), "banner have not been created");

        return rmpBannerConverter.toBaseRmpBannerResponse(banner);
    }

    @ApiOperation(
            httpMethod = "PATCH",
            value = "Изменить настройки баннеров",
            notes = "Возвращает результат операции"
    )
    @RequestMapping(path = "",
            method = RequestMethod.PATCH,
            consumes = MediaType.APPLICATION_JSON_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    public IntapiValidationResponse updateBanners(
            @ApiParam(value = "client_id", required = true) @RequestParam Long rawClientId,
            @ApiParam(value = "uid", required = true) @RequestParam Long uid,
            @ApiParam(value = "banner_ids", required = true) @RequestParam List<Long> bannerIds,
            @RequestBody UpdateRmpBannerRequest request
    ) {
        ClientId clientId = ClientId.fromLong(rawClientId);
        User operator = rmpControllerHelper.getUser(clientId, uid);

        var operationResult = uacBannerService.updateBanners(
                clientId, operator.getUid(), bannerIds, request.getTrackingUrl(), request.getImpressionUrl());

        return validationResultConversionService.buildValidationResponse(operationResult);
    }

    @ApiOperation(
            httpMethod = "DELETE",
            value = "Удалить или заархивировать баннер для рекламы мобильного приложения"
    )
    @RequestMapping(path = "",
            method = RequestMethod.DELETE,
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    public WebResponse deleteRmpBanner(
            @ApiParam(value = "client_id", required = true) @RequestParam Long rawClientId,
            @ApiParam(value = "uid", required = true) @RequestParam Long uid,
            @ApiParam(value = "banner_id", required = true) @RequestParam Long bannerId
    ) {
        ClientId clientId = ClientId.fromLong(rawClientId);
        User operator = rmpControllerHelper.getUser(clientId, uid);
        rmpBannerValidationService.validateBannerId(operator, clientId, bannerId);

        try {
            var massResults = uacBannerService.deleteAndArchiveBanners(operator.getUid(), clientId, Set.of(bannerId));
            var result = massResults.getDeleteMassResult().getResult().isEmpty()
                    ? massResults.getArchiveMassResult()
                    : massResults.getDeleteMassResult();
            if (result.getSuccessfulCount() != 1) {
                throw new IntApiException(
                        HttpStatus.BAD_REQUEST, validationResultConversionService.buildValidationResponse(result));
            }
        } catch (IllegalArgumentException e) {
            throw new IntApiException(HttpStatus.BAD_REQUEST, e.getMessage());
        }

        return new RmpBannerResponse().withId(bannerId);
    }

    @ApiOperation(
            httpMethod = "POST",
            value = "Отправить банеры на модерацию"
    )
    @RequestMapping(path = "moderate",
            method = RequestMethod.POST,
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    public List<IntapiValidationResponse> moderateBanners(
            @ApiParam(value = "client_id", required = true) @RequestParam Long rawClientId,
            @ApiParam(value = "uid", required = true) @RequestParam Long uid,
            @ApiParam(value = "banner_ids", required = true) @RequestParam List<Long> bannerIds
    ) {
        ClientId clientId = ClientId.fromLong(rawClientId);
        User operator = rmpControllerHelper.getUser(clientId, uid);

        MassResult<Long> result = bannerModerateService.moderateBanners(
                clientId, operator.getUid(), bannerIds);
        return validationResultConversionService.buildMassValidationResponse(result);
    }

    @ApiOperation(
            httpMethod = "POST",
            value = "Остановить / запустить баннеры"
    )
    @RequestMapping(path = "suspendResume",
            method = RequestMethod.POST,
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    public List<IntapiValidationResponse> suspendResumeBanners(
            @ApiParam(value = "client_id", required = true) @RequestParam Long rawClientId,
            @ApiParam(value = "uid", required = true) @RequestParam Long uid,
            @ApiParam(value = "banner_ids", required = true) @RequestParam List<Long> bannerIds,
            @ApiParam(value = "status_show", required = true) @RequestParam Boolean statusShow
    ) {
        ClientId clientId = ClientId.fromLong(rawClientId);
        User operator = rmpControllerHelper.getUser(clientId, uid);

        MassResult<Long> result = uacBannerService
                .suspendResumeBanners(clientId, operator.getUid(), statusShow, bannerIds);

        return validationResultConversionService.buildMassValidationResponse(result);
    }

    private List<BannerWithSystemFields> getBanners(
            Long userId, ClientId clientId, Long adGroupId, List<Long> bannerIds
    ) {
        if (adGroupId != null) {
            return uacBannerService.getMobileAppBannersByAdGroupIds(userId, clientId, Set.of(adGroupId));
        } else {
            return uacBannerService.getMobileAppBanners(userId, clientId, Set.copyOf(bannerIds));
        }
    }

    private void createBanner(
            User user, ClientId clientId, MobileContentCampaign campaign, BannerWithSystemFields banner, Long adGroupId
    ) {
        MassResult<Long> result = uacBannerService.createBanner(
                user.getUid(), clientId, campaign, List.of(banner), adGroupId);
        if (result == null) {
            throw new IntApiException(HttpStatus.BAD_REQUEST, "Can't find adGroup with id: " + adGroupId);
        } else if (result.getSuccessfulCount() != 1) {
            throw new IntApiException(
                    HttpStatus.BAD_REQUEST, validationResultConversionService.buildValidationResponse(result));
        }
    }

    private void createAdGroup(
            User user, Client client, MobileContentCampaign campaign,
            BannerWithSystemFields banner, AddRmpBannerRequest request, List<TargetInterest> targetInterests
    ) {
        var keywords = mapList(request.getKeywords(),
                phrase -> new Keyword()
                        .withPhrase(phrase)
                        .withAutobudgetPriority(Constants.DEFAULT_AUTOBUDGET_PRIORITY)
        );
        MassResult<Long> result = uacAdGroupService.createMobileContentAdGroupWithBanner(
                user.getUid(), client, campaign, banner, request.getRegions(), null,
                keywords, request.getMinusKeywords(),  targetInterests);
        if (result.getSuccessfulCount() != 1) {
            throw new IntApiException(
                    HttpStatus.BAD_REQUEST, validationResultConversionService.buildValidationResponse(result));
        }
    }
}
