package ru.yandex.direct.intapi.entity.rmp.banner;

import java.util.List;
import java.util.Map;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.banner.model.Banner;
import ru.yandex.direct.core.entity.banner.model.BannerCreativeStatusModerate;
import ru.yandex.direct.core.entity.banner.model.BannerFlags;
import ru.yandex.direct.core.entity.banner.model.BannerStatusModerate;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.model.ImageBanner;
import ru.yandex.direct.core.entity.banner.model.MobileAppBanner;
import ru.yandex.direct.core.entity.banner.model.NewMobileContentPrimaryAction;
import ru.yandex.direct.core.entity.banner.model.NewReflectedAttribute;
import ru.yandex.direct.core.entity.banner.model.StatusBannerImageModerate;
import ru.yandex.direct.core.entity.moderationdiag.ModerationDiagDataConverterKt;
import ru.yandex.direct.core.entity.moderationdiag.model.ModerationDiag;
import ru.yandex.direct.core.entity.moderationdiag.model.ModerationDiagData;
import ru.yandex.direct.core.entity.moderationreason.model.ModerationReasonObjectType;
import ru.yandex.direct.core.entity.uac.service.ModerationState;

import static com.google.common.base.Preconditions.checkNotNull;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;

@Service
public class RmpBannerConverter {
    public MobileAppBanner toMobileAppBanner(AddRmpBannerRequest request) {
        return new MobileAppBanner()
                .withFlags(
                        request.getAgeLimit() == null
                                ? null
                                : new BannerFlags().with(BannerFlags.AGE, request.getAgeLimit()))
                .withPrimaryAction(NewMobileContentPrimaryAction.GET)
                .withTitle(checkNotNull(request.getTitle()))
                .withBody(checkNotNull(request.getText()))
                .withHref(request.getTrackingUrl())
                .withImpressionUrl(request.getImpressionUrl())
                .withImageHash(request.getImageHash())
                .withCreativeId(request.getCreativeId())
                .withReflectedAttributes(Map.of(
                        NewReflectedAttribute.PRICE, true,
                        NewReflectedAttribute.ICON, true,
                        NewReflectedAttribute.RATING, true,
                        NewReflectedAttribute.RATING_VOTES, true));
    }

    public ImageBanner toImageBanner(AddRmpBannerRequest request) {
        return new ImageBanner()
                .withFlags(
                        request.getAgeLimit() == null
                                ? null
                                : new BannerFlags().with(BannerFlags.AGE, request.getAgeLimit()))
                .withHref(request.getTrackingUrl())
                .withCreativeId(request.getCreativeId())
                .withIsMobileImage(true);
    }

    public RmpBannerResponse toBaseRmpBannerResponse(MobileAppBanner banner) {
        return new RmpBannerResponse()
                .withId(banner.getId())
                .withAdGroupId(banner.getAdGroupId())
                .withCampaignId(banner.getCampaignId())
                .withTitle(banner.getTitle())
                .withText(banner.getBody())
                .withTrackingUrl(banner.getHref())
                .withImpressionUrl(banner.getImpressionUrl())
                .withImageHash(banner.getImageHash())
                .withCreativeId(banner.getCreativeId())
                .withStatusShow(banner.getStatusShow())
                .withStatus(toBannerStatus(banner.getStatusModerate()))
                .withImageStatus(toBannerImageStatus(banner.getImageStatusModerate()))
                .withCreativeStatus(toBannerCreativeStatus(banner.getCreativeStatusModerate()))
                .withState(toBannerState(banner))
                .withFlags(ifNotNull(banner.getFlags(), BannerFlags::getFlags));
    }

    public RmpBannerResponse toBaseRmpBannerResponse(ImageBanner banner) {
        return new RmpBannerResponse()
                .withId(banner.getId())
                .withAdGroupId(banner.getAdGroupId())
                .withCampaignId(banner.getCampaignId())
                .withTrackingUrl(banner.getHref())
                .withCreativeId(banner.getCreativeId())
                .withStatusShow(banner.getStatusShow())
                .withStatus(toBannerStatus(banner.getStatusModerate()))
                .withCreativeStatus(toBannerCreativeStatus(banner.getCreativeStatusModerate()))
                .withState(toBannerState(banner))
                .withFlags(ifNotNull(banner.getFlags(), BannerFlags::getFlags));
    }

    public RmpBannerResponse toBaseRmpBannerResponse(Banner banner) {
        if (banner instanceof MobileAppBanner) {
            return toBaseRmpBannerResponse((MobileAppBanner) banner);
        } else {
            return toBaseRmpBannerResponse((ImageBanner) banner);
        }
    }

    public List<ModerationDiagData> toModerationDiagData(List<ModerationDiag> moderationDiags) {
        return moderationDiags == null
                ? null
                : StreamEx.of(moderationDiags).map(ModerationDiagDataConverterKt::convert).toList();
    }

    public ModerationData toModerationData(Map<ModerationReasonObjectType, List<ModerationDiag>> reasons) {
        return new ModerationData()
                .withBanner(toModerationDiagData(reasons.get(ModerationReasonObjectType.BANNER)))
                .withImage(toModerationDiagData(reasons.get(ModerationReasonObjectType.IMAGE)))
                .withCreative(toModerationDiagData(reasons.get(ModerationReasonObjectType.VIDEO_ADDITION)));
    }

    public List<RmpBannerResponse> toRmpBannerResponses(
            List<BannerWithSystemFields> banners,
            Map<Long, Map<ModerationReasonObjectType, List<ModerationDiag>>> moderationDiagsById) {
        Map<Long, ModerationData> moderationDataById = EntryStream.of(moderationDiagsById)
                .mapValues(this::toModerationData)
                .toMap();
        return StreamEx.of(banners)
                .map(banner -> toBaseRmpBannerResponse(banner)
                        .withModerationData(moderationDataById.get(banner.getId())))
                .toList();
    }

    public BannerState toBannerState(MobileAppBanner banner) {
        if (banner.getStatusArchived()) {
            return BannerState.ARCHIVED;
        }
        if (!banner.getStatusShow()) {
            return BannerState.SUSPENDED;
        }
        if (banner.getStatusActive()) {
            return BannerState.ON;
        }
        return BannerState.OFF;
    }

    public BannerState toBannerState(ImageBanner banner) {
        if (banner.getStatusArchived()) {
            return BannerState.ARCHIVED;
        }
        if (!banner.getStatusShow()) {
            return BannerState.SUSPENDED;
        }
        if (banner.getStatusActive()) {
            return BannerState.ON;
        }
        return BannerState.OFF;
    }

    public ModerationState toBannerStatus(BannerStatusModerate status) {
        if (status == null) {
            return null;
        }
        switch (status) {
            case YES:
                return ModerationState.ACCEPTED;
            case NO:
                return ModerationState.REJECTED;
            case NEW:
                return ModerationState.DRAFT;
            case SENT:
                return ModerationState.MODERATION;
            case READY:
                return ModerationState.READY;
            default:
                return null;
        }
    }

    public ModerationState toBannerImageStatus(StatusBannerImageModerate status) {
        if (status == null) {
            return null;
        }
        switch (status) {
            case YES:
                return ModerationState.ACCEPTED;
            case NO:
                return ModerationState.REJECTED;
            case NEW:
                return ModerationState.DRAFT;
            case READY:
                return ModerationState.READY;
            case SENT:
            case SENDING:
                return ModerationState.MODERATION;
            default:
                return null;
        }
    }

    public ModerationState toBannerCreativeStatus(BannerCreativeStatusModerate status) {
        if (status == null) {
            return null;
        }
        switch (status) {
            case YES:
                return ModerationState.ACCEPTED;
            case NO:
                return ModerationState.REJECTED;
            case NEW:
                return ModerationState.DRAFT;
            case READY:
                return ModerationState.READY;
            case SENT:
            case SENDING:
                return ModerationState.MODERATION;
            default:
                return null;
        }
    }
}
