package ru.yandex.direct.intapi.entity.rmp.campaigns;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.LocalDate;
import java.util.Collections;
import java.util.EnumSet;
import java.util.List;
import java.util.Map;

import one.util.streamex.StreamEx;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.campaign.model.CampOptionsStrategy;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CampaignsAutobudget;
import ru.yandex.direct.core.entity.campaign.model.DayBudgetShowMode;
import ru.yandex.direct.core.entity.campaign.model.DbStrategy;
import ru.yandex.direct.core.entity.campaign.model.MobileContentCampaign;
import ru.yandex.direct.core.entity.campaign.model.SmsFlag;
import ru.yandex.direct.core.entity.campaign.model.StrategyData;
import ru.yandex.direct.core.entity.campaign.model.StrategyName;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.mobileapp.model.MobileApp;
import ru.yandex.direct.core.entity.mobileapp.model.MobileAppDeviceTypeTargeting;
import ru.yandex.direct.core.entity.mobileapp.model.MobileAppNetworkTargeting;
import ru.yandex.direct.core.entity.mobileapp.model.MobileAppTracker;
import ru.yandex.direct.core.entity.mobileapp.model.MobileAppTrackerTrackingSystem;
import ru.yandex.direct.core.entity.mobileapp.service.MobileAppService;
import ru.yandex.direct.core.entity.mobilecontent.model.AgeLabel;
import ru.yandex.direct.core.entity.mobilecontent.model.MobileContent;
import ru.yandex.direct.core.entity.mobilecontent.model.MobileContentAvatarSize;
import ru.yandex.direct.core.entity.mobilecontent.model.StatusIconModerate;
import ru.yandex.direct.core.entity.moderationdiag.ModerationDiagDataConverterKt;
import ru.yandex.direct.core.entity.moderationdiag.model.ModerationDiag;
import ru.yandex.direct.core.entity.moderationdiag.model.ModerationDiagData;
import ru.yandex.direct.core.entity.moderationreason.model.ModerationReasonObjectType;
import ru.yandex.direct.core.entity.uac.converter.UacCampaignConverter;
import ru.yandex.direct.core.entity.uac.service.ModerationState;
import ru.yandex.direct.core.entity.uac.service.UacCampaignStates;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.libs.timetarget.TimeTargetUtils;

import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.DEFAULT_ENABLE_CPC_HOLD;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.DEFAULT_SMS_TIME_INTERVAL;
import static ru.yandex.direct.core.entity.mobilecontent.service.MobileContentService.generateUrlString;
import static ru.yandex.direct.utils.CommonUtils.nullableNvl;
import static ru.yandex.direct.utils.CommonUtils.nvl;

@Service
public class RmpCampaignConverter {
    private static final BigDecimal MONEY_MULTIPLIER = BigDecimal.valueOf(1_000_000);

    private final MobileAppService mobileAppService;
    private final FeatureService featureService;


    public RmpCampaignConverter(MobileAppService mobileAppService, FeatureService featureService) {
        this.mobileAppService = mobileAppService;
        this.featureService = featureService;
    }

    public BigDecimal toCoreMoney(Long rmpMoney) {
        return BigDecimal.valueOf(rmpMoney).divide(MONEY_MULTIPLIER, 6, RoundingMode.HALF_UP);
    }

    public Long toRmpMoney(BigDecimal coreMoney) {
        return coreMoney.multiply(MONEY_MULTIPLIER).setScale(0, RoundingMode.HALF_UP).longValue();
    }

    private StrategyData toStrategyData(StrategyName strategyName, AddRmpCampaignRequest request) {
        StrategyData strategyData = new StrategyData()
                .withName(StrategyName.toSource(strategyName).getLiteral())
                .withSum(toCoreMoney(request.getWeekCost()))
                .withVersion(1L);
        switch (strategyName) {
            case AUTOBUDGET_AVG_CPA:
                return strategyData
                        .withPayForConversion(true)
                        .withAvgCpa(toCoreMoney(request.getCost()))
                        .withGoalId(request.getTarget().getGoalId());
            case AUTOBUDGET_AVG_CPI:
                return strategyData
                        .withPayForConversion(request.getPayForConversion())
                        .withAvgCpi(toCoreMoney(request.getCost()))
                        .withGoalId(request.getTarget().getGoalId());
            case AUTOBUDGET_AVG_CLICK:
                return strategyData
                        .withAvgBid(toCoreMoney(request.getCost()));
            default:
                throw new IllegalStateException("Illegal Strategy name " + strategyName);
        }
    }

    private DbStrategy toStrategy(AddRmpCampaignRequest request, ClientId clientId) {
        StrategyName strategyName;
        if (request.getTarget() == Target.CPC || nvl(request.getIsSkadNetworkEnabled(), false)) {
            strategyName = StrategyName.AUTOBUDGET_AVG_CLICK;
        } else if (featureService.isEnabledForClientId(clientId, FeatureName.UAC_FIX_CPA_STRATEGY_ENABLED)) {
            strategyName = StrategyName.AUTOBUDGET_AVG_CPA;
        } else {
            strategyName = StrategyName.AUTOBUDGET_AVG_CPI;
        }
        return (DbStrategy) new DbStrategy()
                .withStrategy(CampOptionsStrategy.DIFFERENT_PLACES)
                .withPlatform(UacCampaignConverter.toCampaignsPlatform(request.getStrategyPlatform()))
                .withStrategyName(strategyName)
                .withAutobudget(CampaignsAutobudget.YES)
                .withStrategyData(toStrategyData(strategyName, request));
    }

    public MobileContentCampaign toCampaign(AddRmpCampaignRequest request, ClientId clientId) {
        return new MobileContentCampaign()
                .withType(CampaignType.MOBILE_CONTENT)
                .withName(request.getName())
                .withGeo(Collections.emptySet())
                .withEnableCpcHold(DEFAULT_ENABLE_CPC_HOLD)
                .withDeviceTypeTargeting(EnumSet.allOf(MobileAppDeviceTypeTargeting.class))
                .withDayBudgetShowMode(DayBudgetShowMode.DEFAULT_)
                .withHasExtendedGeoTargeting(false)
                .withSmsFlags(EnumSet.noneOf(SmsFlag.class))
                .withNetworkTargeting(EnumSet.allOf(MobileAppNetworkTargeting.class))
                .withSmsTime(DEFAULT_SMS_TIME_INTERVAL)
                .withStartDate(LocalDate.now())
                .withTimeZoneId(TimeTargetUtils.DEFAULT_TIMEZONE)
                .withDayBudget(BigDecimal.valueOf(0L))
                .withIsSkadNetworkEnabled(request.getIsSkadNetworkEnabled())
                .withIsAllowedOnAdultContent(request.getAdultContentEnabled())
                .withStrategy(toStrategy(request, clientId))
                .withSource(request.getSource())
                .withHasExtendedGeoTargeting(true);
    }

    public MobileApp toMobileApp(AddAppInfoRequest request) {
        Long mobileAppId = request.getId();
        if (mobileAppId != null) {
            return new MobileApp().withId(mobileAppId);
        }
        MobileApp mobileApp = new MobileApp()
                            .withName(request.getName())
                            .withStoreHref(request.getUrl())
                            .withDisplayedAttributes(Collections.emptySet());
        if (request.getTrackerUrl() == null) {
            return mobileApp.withTrackers(Collections.emptyList());
        }
        return mobileApp
                .withTrackers(List.of(
                        new MobileAppTracker()
                            .withTrackingSystem(nvl(request.getTrackingSystem(), MobileAppTrackerTrackingSystem.OTHER))
                            .withUrl(request.getTrackerUrl())
                            .withImpressionUrl(request.getImpressionUrl())
                            .withUserParams(Collections.emptyList())
                ));
    }

    public RmpCampaignResponse toBaseCampaignResponse(MobileContentCampaign campaign, UacCampaignStates states) {
        ClientId clientId = ClientId.fromLong(campaign.getClientId());
        MobileApp mobileApp = mobileAppService.getMobileApp(clientId, campaign.getMobileAppId()).get();
        Long campaignId = campaign.getId();
        StrategyData strategyData = campaign.getStrategy().getStrategyData();
        return new RmpCampaignResponse()
                .withId(campaignId)
                .withTitle(campaign.getName())
                .withAppTitle(mobileApp.getName())
                .withStoreUrl(mobileApp.getStoreHref())
                .withIconUrl(toIconUrl(mobileApp.getMobileContent()))
                .withRating(mobileApp.getMobileContent().getRating())
                .withRatingVotes(mobileApp.getMobileContent().getRatingVotes())
                .withAgeLimit(toAgeLimit(mobileApp.getMobileContent().getAgeLabel()))
                .withMinOs(mobileApp.getMobileContent().getMinOsVersion())
                .withRegions(campaign.getGeo())
                .withTarget(Target.getByGoalId(strategyData.getGoalId()))
                .withCost(toRmpMoney(nvl(strategyData.getAvgCpi(),
                        nullableNvl(strategyData.getAvgBid(), strategyData.getAvgCpa()))))
                .withWeekLimit(toRmpMoney(strategyData.getSum()))
                .withIconState(toIconState(mobileApp.getMobileContent().getStatusIconModerate()))
                .withStatusShow(nvl(campaign.getStatusShow(), true))
                .withCampaignStatus(states.getModerationState())
                .withCampaignState(states.getCampaignState())
                .withCampaignStateReasons(states.getCampaignStateReasons());
    }

    public List<ModerationDiagData> toModerationDiagData(List<ModerationDiag> moderationDiags) {
        return moderationDiags == null
                ? null
                : StreamEx.of(moderationDiags).map(ModerationDiagDataConverterKt::convert).toList();
    }

    public RmpCampaignResponse toCampaignResponse(
            MobileContentCampaign campaign,
            UacCampaignStates campaignStates,
            Map<ModerationReasonObjectType, List<ModerationDiag>> moderationDiagsByReason
    ) {
        List<ModerationDiagData> moderationDiags = moderationDiagsByReason == null
                ? null
                : toModerationDiagData(moderationDiagsByReason.get(ModerationReasonObjectType.CAMPAIGN));
        return toBaseCampaignResponse(campaign, campaignStates)
                .withModerationDiags(moderationDiags);

    }

    private String toIconUrl(MobileContent mobileContent) {
        if (mobileContent.getIconHash() == null || mobileContent.getOsType() == null) {
            return null;
        }
        return generateUrlString(
                "avatars.mds.yandex.net",
                mobileContent.getOsType(),
                mobileContent.getIconHash(),
                MobileContentAvatarSize.ICON);
    }

    private ModerationState toIconState(StatusIconModerate status) {
        if (status == null) {
            return null;
        }
        switch (status) {
            case YES:
                return ModerationState.ACCEPTED;
            case NO:
                return ModerationState.REJECTED;
            case SENDING:
                return ModerationState.DRAFT;
            case SENT:
                return ModerationState.MODERATION;
            case READY:
                return ModerationState.READY;
            default:
                return null;
        }
    }

    private Long toAgeLimit(AgeLabel ageLabel) {
        if (ageLabel == null) {
            return null;
        }
        switch (ageLabel) {
            case _0_2B:
                return 0L;
            case _6_2B:
                return 6L;
            case _12_2B:
                return 12L;
            case _16_2B:
                return 16L;
            case _18_2B:
                return 18L;
            default:
                return null;
        }
    }
}
