package ru.yandex.direct.intapi.entity.rmp.campaigns;

import java.util.List;
import java.util.Map;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.direct.core.entity.campaign.model.MobileContentCampaign;
import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.client.service.ClientGeoService;
import ru.yandex.direct.core.entity.mobileapp.model.MobileApp;
import ru.yandex.direct.core.entity.moderationdiag.model.ModerationDiag;
import ru.yandex.direct.core.entity.moderationreason.model.ModerationReasonObjectType;
import ru.yandex.direct.core.entity.moderationreason.service.ModerationReasonService;
import ru.yandex.direct.core.entity.region.validation.RegionIdsValidator;
import ru.yandex.direct.core.entity.uac.model.UpdateUacCampaignRequest;
import ru.yandex.direct.core.entity.uac.service.RmpCampaignService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.model.UidAndClientId;
import ru.yandex.direct.intapi.IntApiException;
import ru.yandex.direct.intapi.entity.rmp.RmpControllerHelper;
import ru.yandex.direct.intapi.validation.kernel.ValidationResultConversionService;
import ru.yandex.direct.intapi.validation.model.IntapiValidationResponse;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.tvm.AllowServices;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.web.core.model.WebResponse;

import static ru.yandex.direct.tvm.TvmService.DIRECT_DEVELOPER;
import static ru.yandex.direct.tvm.TvmService.RMP_PROD;
import static ru.yandex.direct.tvm.TvmService.RMP_TEST;


@RestController
@Api(value = "Контроллер для управления кампаниями для рекламы мобильных приложений")
@RequestMapping(value = "/rmp/v1/campaigns")
@AllowServices(production = {RMP_PROD}, testing = {RMP_TEST, DIRECT_DEVELOPER})
public class RmpCampaignsController {
    private final RmpCampaignConverter rmpCampaignConverter;
    private final RmpControllerHelper rmpControllerHelper;
    private final ValidationResultConversionService validationResultConversionService;
    private final ClientGeoService clientGeoService;
    private final RegionIdsValidator regionIdsValidator;
    private final ModerationReasonService moderationReasonService;

    private final RmpCampaignService rmpCampaignService;

    @Autowired
    @SuppressWarnings("checkstyle:parameternumber")
    public RmpCampaignsController(
            RmpCampaignConverter rmpCampaignConverter,
            RmpControllerHelper rmpControllerHelper,
            ValidationResultConversionService validationResultConversionService,
            ClientGeoService clientGeoService,
            ModerationReasonService moderationReasonService,
            RmpCampaignService rmpCampaignService) {
        this.rmpCampaignConverter = rmpCampaignConverter;
        this.rmpControllerHelper = rmpControllerHelper;
        this.validationResultConversionService = validationResultConversionService;
        this.clientGeoService = clientGeoService;
        this.moderationReasonService = moderationReasonService;
        this.rmpCampaignService = rmpCampaignService;
        this.regionIdsValidator = new RegionIdsValidator();
    }

    @ApiOperation(
            httpMethod = "GET",
            value = "Получить список кампаний для рекламы мобильных приложений",
            notes = "Возвращает список кампаний"
    )
    @RequestMapping(path = "",
            method = RequestMethod.GET,
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    public List<RmpCampaignResponse> getRmpCampaigns(
            @ApiParam(value = "id", required = true) @RequestParam Long campaignId,
            @ApiParam(value = "client_id", required = true) @RequestParam Long rawClientId
    ) {
        ClientId clientId = ClientId.fromLong(rawClientId);
        rmpControllerHelper.getClient(clientId);

        MobileContentCampaign campaign = rmpCampaignService.getMobileContentCampaign(clientId, campaignId);
        if (campaign == null) {
            throw new IntApiException(HttpStatus.BAD_REQUEST, "Campaign not found");
        }
        var moderationDiagsByReason = getModerationDiagsByReason(clientId, campaignId);
        var campaignStates = rmpCampaignService.getCampaignStates(clientId, campaign);
        return List.of(rmpCampaignConverter.toCampaignResponse(campaign, campaignStates, moderationDiagsByReason));
    }

    @ApiOperation(
            httpMethod = "POST",
            value = "Создать кампанию для рекламы мобильного приложения",
            notes = "Возвращает id созданной кампании"
    )
    @RequestMapping(path = "",
            method = RequestMethod.POST,
            consumes = MediaType.APPLICATION_JSON_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    public WebResponse addRmpCampaign(
            @ApiParam(value = "client_id", required = true) @RequestParam Long rawClientId,
            @ApiParam(value = "uid", required = true) @RequestParam Long uid,
            @RequestBody AddRmpCampaignRequest request
    ) {
        ClientId clientId = ClientId.fromLong(rawClientId);
        Client client = rmpControllerHelper.getClient(clientId);
        User user = rmpControllerHelper.getUser(clientId, uid);
        if (request.getRegions() != null && !request.getRegions().isEmpty()) {
            validateRegions(clientId, request.getRegions());
        }

        MobileApp mobileApp = rmpCampaignConverter.toMobileApp(request.getAppInfo());
        MobileContentCampaign campaign = rmpCampaignConverter.toCampaign(request, clientId)
                .withEmail(user.getEmail())
                .withMobileAppId(null);  // mobileAppId неизвестен до создания мобильного приложения

        Result<MobileContentCampaign> result = rmpCampaignService.addRmpCampaign(user,
                UidAndClientId.of(client.getChiefUid(), clientId), campaign, mobileApp);
        if (!result.isSuccessful()) {
            throw new IntApiException(
                    HttpStatus.BAD_REQUEST,
                    validationResultConversionService.buildValidationResponse(result));
        }
        var campaignStates = rmpCampaignService.getCampaignStates(clientId, campaign);
        return rmpCampaignConverter.toBaseCampaignResponse(result.getResult(), campaignStates);
    }

    @ApiOperation(
            httpMethod = "PATCH",
            value = "Изменить настройки кампании",
            notes = "Возвращает результат операции"
    )
    @RequestMapping(path = "",
            method = RequestMethod.PATCH,
            consumes = MediaType.APPLICATION_JSON_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    public IntapiValidationResponse updateRmpCampaign(
            @ApiParam(value = "client_id", required = true) @RequestParam Long rawClientId,
            @ApiParam(value = "uid", required = true) @RequestParam Long uid,
            @ApiParam(value = "id", required = true) @RequestParam Long campaignId,
            @RequestBody UpdateUacCampaignRequest request
    ) {
        ClientId clientId = ClientId.fromLong(rawClientId);
        Client client = rmpControllerHelper.getClient(clientId);
        User operator = rmpControllerHelper.getUser(clientId, uid);

        MobileContentCampaign campaign = rmpCampaignService.getMobileContentCampaign(clientId, campaignId);
        if (campaign == null) {
            throw new IntApiException(HttpStatus.BAD_REQUEST, "Campaign not found");
        }

        var campaignResult = rmpCampaignService.updateRmpCampaign(client, operator, campaign, request);

        return validationResultConversionService.buildValidationResponse(campaignResult);
    }

    @ApiOperation(
            httpMethod = "DELETE",
            value = "Удалить кампанию для рекламы мобильного приложения"
    )
    @RequestMapping(path = "",
            method = RequestMethod.DELETE,
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    public IntapiValidationResponse deleteRmpCampaign(
            @ApiParam(value = "client_id", required = true) @RequestParam Long rawClientId,
            @ApiParam(value = "uid", required = true) @RequestParam Long uid,
            @ApiParam(value = "campaign_id", required = true) @RequestParam Long campaignId
    ) {
        ClientId clientId = ClientId.fromLong(rawClientId);
        User operator = rmpControllerHelper.getUser(clientId, uid);

        MassResult<Long> result = rmpCampaignService.deleteCampaign(clientId, operator.getUid(), campaignId);

        return validationResultConversionService.buildValidationResponse(result);
    }

    public Map<ModerationReasonObjectType, List<ModerationDiag>> getModerationDiagsByReason(
            ClientId clientId, Long campaignId) {
        var campaignIdsByReason = Map.of(
                ModerationReasonObjectType.CAMPAIGN, List.of(campaignId)
        );

        return moderationReasonService
                .getRejectReasonDiags(clientId, campaignIdsByReason).get(campaignId);
    }

    void validateRegions(ClientId clientId, List<Long> regions) {
        GeoTree geoTree = clientGeoService.getClientTranslocalGeoTree(clientId);
        ValidationResult<List<Long>, Defect> validation = regionIdsValidator.apply(regions, geoTree);
        if (validation.hasAnyErrors()) {
            throw new IntApiException(
                    HttpStatus.BAD_REQUEST, validationResultConversionService.buildValidationResponse(validation));
        }
    }
}
