package ru.yandex.direct.intapi.entity.rmp.content;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.direct.core.entity.image.container.UploadedBannerImageInformation;
import ru.yandex.direct.core.entity.image.model.BannerImageSource;
import ru.yandex.direct.core.entity.image.service.ImageService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.intapi.IntApiException;
import ru.yandex.direct.intapi.validation.kernel.ValidationResultConversionService;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.tvm.AllowServices;
import ru.yandex.direct.web.core.model.WebResponse;

import static ru.yandex.direct.core.validation.ValidationUtils.hasValidationIssues;
import static ru.yandex.direct.tvm.TvmService.DIRECT_DEVELOPER;
import static ru.yandex.direct.tvm.TvmService.RMP_PROD;
import static ru.yandex.direct.tvm.TvmService.RMP_TEST;


@RestController
@Api(value = "Контроллер для создания контентов для объявлений о мобильных приложениях")
@RequestMapping(value = "/rmp/v1/content")
@AllowServices(production = {RMP_PROD}, testing = {RMP_TEST, DIRECT_DEVELOPER})
public class RmpContentController {
    private final ImageService imageService;
    private final RmpContentConverter rmpContentConverter;
    private final ValidationResultConversionService validationResultConversionService;

    @Autowired
    public RmpContentController(
            ImageService imageService,
            RmpContentConverter rmpContentConverter,
            ValidationResultConversionService validationResultConversionService
    ) {
        this.imageService = imageService;
        this.rmpContentConverter = rmpContentConverter;
        this.validationResultConversionService = validationResultConversionService;
    }

    @ApiOperation(
            httpMethod = "POST",
            value = "Создать контент для рекламы мобильного приложения",
            notes = "Возвращает документ с информацией о созданном контенте"
    )
    @RequestMapping(path = "",
            method = RequestMethod.POST,
            consumes = MediaType.APPLICATION_JSON_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    public WebResponse addRmpContent(
            @ApiParam(value = "client_id", required = true) @RequestParam Long rawClientId,
            @RequestBody AddRmpContentRequest request
    ) {
        ClientId clientId = ClientId.fromLong(rawClientId);

        BannerImageSource source = rmpContentConverter.toBannerImageSource(request.getIsUac());

        @SuppressWarnings("ConstantConditions")
        Result<UploadedBannerImageInformation> operationResult = imageService.saveImageFromUrl(
                clientId,
                request.getUrl(),
                request.getBannerImageType(),
                source,
                null);
        if (hasValidationIssues(operationResult)) {
            throw new IntApiException(
                    HttpStatus.BAD_REQUEST, validationResultConversionService.buildValidationResponse(operationResult));
        }
        return rmpContentConverter.toRmpContentResponse(operationResult.getResult());
    }
}
