package ru.yandex.direct.intapi.entity.showconditions.model.request;

import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;

import com.fasterxml.jackson.annotation.JsonProperty;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.BooleanUtils;
import org.apache.commons.lang3.tuple.Pair;

import ru.yandex.direct.common.logging.RequestWithCampaignId;
import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.core.entity.relevancematch.model.RelevanceMatch;
import ru.yandex.direct.core.entity.retargeting.model.Retargeting;
import ru.yandex.direct.model.ModelChanges;

import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.core.entity.keyword.service.KeywordUtils.hasAutotargetingPrefix;
import static ru.yandex.direct.core.entity.keyword.service.KeywordUtils.phraseWithoutAutotargetingPrefix;

/**
 * Запрос на изменение условий нацеливания
 * При добавлении нового условия нужно добавить получение id групп в метод {@code getAdGroupIds}
 */
public class ShowConditionsRequest implements RequestWithCampaignId {
    private Map<Long, RetargetingModificationContainer> retargetings = new HashMap<>();

    private Map<Long, RetargetingModificationContainer> searchRetargetings = new HashMap<>();

    private Map<Long, KeywordModificationContainer> keywords = new HashMap<>();

    private Map<Long, RetargetingModificationContainer> targetInterests = new HashMap<>();

    private Map<Long, RelevanceMatchModificationContainer> relevanceMatches = new HashMap<>();

    @JsonProperty("copy_group_if_oversized")
    private boolean copyOversizeGroupFlag;

    public Map<Long, RetargetingModificationContainer> getRetargetings() {
        return retargetings;
    }

    public ShowConditionsRequest withRetargetings(Map<Long, RetargetingModificationContainer> retargetings) {
        this.retargetings = retargetings;
        return this;
    }

    public Map<Long, RetargetingModificationContainer> getSearchRetargetings() {
        return searchRetargetings;
    }

    public ShowConditionsRequest withSearchRetargetings(
            Map<Long, RetargetingModificationContainer> searchRetargetings
    ) {
        this.searchRetargetings = searchRetargetings;
        return this;
    }

    public Map<Long, KeywordModificationContainer> getKeywords() {
        return keywords;
    }

    public ShowConditionsRequest withKeywords(Map<Long, KeywordModificationContainer> keywords) {
        this.keywords = keywords;
        return this;
    }

    public Map<Long, RetargetingModificationContainer> getTargetInterests() {
        return targetInterests;
    }

    public ShowConditionsRequest withTargetInterests(Map<Long, RetargetingModificationContainer> targetInterests) {
        this.targetInterests = targetInterests;
        return this;
    }

    public Map<Long, RelevanceMatchModificationContainer> getRelevanceMatches() {
        return this.relevanceMatches;
    }

    public ShowConditionsRequest withRelevanceMatches(
            final Map<Long, RelevanceMatchModificationContainer> relevanceMatches) {
        this.relevanceMatches = relevanceMatches;
        return this;
    }

    public boolean getCopyOversizeGroupFlag() {
        return copyOversizeGroupFlag;
    }

    public ShowConditionsRequest withCopyOversizeGroupFlag(boolean copyOversizeGroupFlag) {
        this.copyOversizeGroupFlag = copyOversizeGroupFlag;
        return this;
    }

    /**
     * @return список пар: rowHash, ключевая фраза для добавления.
     * Rowhash нужен в последствии для логирования в ОКР.
     */
    public List<Pair<String, Keyword>> getKeywordsToAdd() {
        return EntryStream.of(keywords)
                .mapValues(KeywordModificationContainer::getAdded)
                .flatMapValues(List::stream)
                .map(itm -> Pair.of(itm.getValue().getReportRowHash(),
                        new Keyword()
                                .withAdGroupId(itm.getKey())
                                .withPhrase(phraseWithoutAutotargetingPrefix(itm.getValue().getPhrase()))
                                .withIsAutotargeting(hasAutotargetingPrefix(itm.getValue().getPhrase()))
                                .withPrice(itm.getValue().getPrice())
                                .withPriceContext(itm.getValue().getPriceContext())
                                .withAutobudgetPriority(itm.getValue().getAutobudgetPriority())))
                .collect(toList());
    }

    /**
     * @return список идентификаторов ключевых фраз для удаления
     * левая часть {@link Pair} - id группы, правая - id фразы
     */
    public List<Pair<Long, Long>> getAdGroupIdToKeywordToDelete() {
        return EntryStream.of(keywords)
                .mapValues(KeywordModificationContainer::getDeleted)
                .flatMapValues(List::stream)
                .map(entry -> Pair.of(entry.getKey(), entry.getValue()))
                .toList();
    }


    /**
     * @return список изменений ключевых фраз, левая часть {@link Pair} - id группы, правая - {@link ModelChanges}
     */
    public List<Pair<Long, ModelChanges<Keyword>>> getKeywordChangesToEdit() {
        return EntryStream.of(keywords)
                .mapValues(KeywordModificationContainer::getUpdated)
                .mapValues(Map::entrySet)
                .flatMapValues(Collection::stream)
                .mapValues(e -> {
                    ModelChanges<Keyword> changes =
                            new ModelChanges<>(e.getKey(), Keyword.class)
                                    .processNotNull(e.getValue().getPhrase(), Keyword.PHRASE)
                                    .processNotNull(e.getValue().getPrice(), Keyword.PRICE)
                                    .processNotNull(e.getValue().getPriceContext(), Keyword.PRICE_CONTEXT)
                                    .processNotNull(e.getValue().getSuspended(), Keyword.IS_SUSPENDED,
                                            BooleanUtils::toBooleanObject);
                    Integer priority = e.getValue().getAutobudgetPriority();
                    if (priority != null && priority > 0) {
                        changes.process(priority, Keyword.AUTOBUDGET_PRIORITY);
                    }
                    return changes;
                })
                .map(entry -> Pair.of(entry.getKey(), entry.getValue()))
                .toList();
    }

    /**
     * @return список идентификаторов ретаргетингов для удаления
     */
    public List<Long> getRetargetingIdsToDelete() {
        return getIdsToDelete(retargetings);
    }

    /**
     * S
     *
     * @return список изменений ретаргетингов
     */
    public List<ModelChanges<Retargeting>> getRetargetingChangesToEdit() {
        return getChangesToEdit(retargetings);
    }

    public Map<Long, Long> retargetingIdToAdGroupIdMap() {
        return idToAdGroupIdMap(retargetings);
    }

    public List<Long> getSearchRetargetingIdsToDelete() {
        return getIdsToDelete(searchRetargetings);
    }

    public Map<Long, Long> searchRetargetingsToAdGroupIdMap() {
        return idToAdGroupIdMap(searchRetargetings);
    }

    /**
     * @return список идентификаторов таргетов по интересам для удаления
     */
    public List<Long> getTargetInterestsIdsToDelete() {
        return getIdsToDelete(targetInterests);
    }

    /**
     * @return список изменений таргетов по интересам
     */
    public List<ModelChanges<Retargeting>> getTargetInterestsChangesToEdit() {
        return getChangesToEdit(targetInterests);
    }

    public Map<Long, Long> targetInterestsIdToAdGroupIdMap() {
        return idToAdGroupIdMap(targetInterests);
    }

    /**
     * @return список идентификаторов таргетов по интересам и ретаргетингов для удаления
     */
    private List<Long> getIdsToDelete(Map<Long, RetargetingModificationContainer> container) {
        return container.values().stream()
                .map(RetargetingModificationContainer::getDeleted)
                .flatMap(List::stream)
                .collect(toList());
    }

    /**
     * @return список изменений таргетов по интересам и ретаргетингов
     */
    private List<ModelChanges<Retargeting>> getChangesToEdit(Map<Long, RetargetingModificationContainer> container) {
        return container.values().stream()
                .map(RetargetingModificationContainer::getEdited)
                .map(Map::entrySet)
                .flatMap(Collection::stream)
                .map(e -> new ModelChanges<>(e.getKey(), Retargeting.class)
                        .processNotNull(e.getValue().getPriceContext(), Retargeting.PRICE_CONTEXT)
                        .processNotNull(e.getValue().getAutobudgetPriority(), Retargeting.AUTOBUDGET_PRIORITY)
                        .processNotNull(e.getValue().getSuspended(), Retargeting.IS_SUSPENDED,
                                BooleanUtils::toBooleanObject)
                )
                .collect(toList());
    }

    private Map<Long, Long> idToAdGroupIdMap(Map<Long, RetargetingModificationContainer> container) {
        Map<Long, Long> map = new HashMap<>();
        container.forEach((adGroupId, e) -> {
            e.getEdited().forEach((id, r) -> map.put(id, adGroupId));
            e.getDeleted().forEach(id -> map.put(id, adGroupId));
        });

        return map;
    }

    /**
     * @return список бесфразных таргетингов для добавления
     */
    public List<RelevanceMatch> getRelevanceMatchesToAdd() {
        return EntryStream.of(relevanceMatches)
                .mapValues(RelevanceMatchModificationContainer::getAdded)
                .flatMapValues(List::stream)
                .map(itm -> new RelevanceMatch()
                        .withAdGroupId(itm.getKey())
                        .withPrice(itm.getValue().getPrice())
                        .withPriceContext(itm.getValue().getPriceContext())
                        .withAutobudgetPriority(itm.getValue().getAutobudgetPriority())
                )
                .collect(toList());
    }

    /**
     * @return список изменений бесфразных таргетингов с указанием номера группы слева
     */
    public List<Pair<Long, ModelChanges<RelevanceMatch>>> getRelevanceMatchChangesToEdit() {
        Function<Map.Entry<Long, RelevanceMatchItem>, ModelChanges<RelevanceMatch>> entryToModelChanges = entry -> {
            ModelChanges<RelevanceMatch> changes = new ModelChanges<>(entry.getKey(), RelevanceMatch.class)
                    .processNotNull(entry.getValue().getPrice(), RelevanceMatch.PRICE)
                    .processNotNull(entry.getValue().getPriceContext(), RelevanceMatch.PRICE_CONTEXT)
                    .processNotNull(entry.getValue().getIsSuspended(), RelevanceMatch.IS_SUSPENDED,
                            BooleanUtils::toBooleanObject);
            Integer priority = entry.getValue().getAutobudgetPriority();
            if (priority != null && priority > 0) {
                changes.process(priority, RelevanceMatch.AUTOBUDGET_PRIORITY);
            }

            return changes;
        };

        return EntryStream.of(relevanceMatches)
                .mapValues(RelevanceMatchModificationContainer::getEdited)
                .mapValues(Map::entrySet)
                .flatMapValues(Collection::stream)
                .mapValues(entryToModelChanges)
                .map(entry -> Pair.of(entry.getKey(), entry.getValue()))
                .toList();
    }


    /**
     * @return список идентификаторов бесфразных таргетингов для удаления
     * левая часть {@link Pair} - id группы, правая - id бесфразного таргетинга
     */
    public List<Pair<Long, Long>> getAdGroupIdToRelevanceMatchToDelete() {
        return EntryStream.of(relevanceMatches)
                .mapValues(RelevanceMatchModificationContainer::getDeleted)
                .flatMapValues(List::stream)
                .map(entry -> Pair.of(entry.getKey(), entry.getValue()))
                .toList();
    }

    @Override
    public Set<Long> getAdGroupIds() {
        return StreamEx.of(keywords.keySet())
                .append(relevanceMatches.keySet())
                .append(retargetings.keySet())
                .append(searchRetargetings.keySet())
                .append(targetInterests.keySet())
                .toSet();
    }

    public List<Long> getMainBids() {
        return StreamEx.of(keywords.values().stream().map(KeywordModificationContainer::getMainBid))
                .append(retargetings.values().stream().map(RetargetingModificationContainer::getMainBid))
                .append(targetInterests.values().stream().map(RetargetingModificationContainer::getMainBid))
                .append(relevanceMatches.values().stream().map(RelevanceMatchModificationContainer::getMainBid))
                .nonNull()
                .toList();

    }
}
