package ru.yandex.direct.intapi.entity.showconditions.model.response;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import one.util.streamex.EntryStream;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.tuple.Pair;

import ru.yandex.direct.core.entity.auction.container.bs.TrafaretBidItem;
import ru.yandex.direct.core.entity.bidmodifiers.service.BidModifierService;
import ru.yandex.direct.core.entity.bids.container.KeywordBidPokazometerData;
import ru.yandex.direct.core.entity.keyword.container.AddedKeywordInfo;
import ru.yandex.direct.core.entity.keyword.container.KeywordsModificationResult;
import ru.yandex.direct.core.entity.keyword.container.StopwordsFixation;
import ru.yandex.direct.core.entity.keyword.container.UpdatedKeywordInfo;
import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.core.entity.keyword.service.KeywordsModifyOperation;
import ru.yandex.direct.core.entity.keyword.service.validation.KeywordDefectIds;
import ru.yandex.direct.core.entity.keyword.service.validation.KeywordsLimitDefectParams;
import ru.yandex.direct.core.entity.relevancematch.container.RelevanceMatchModificationResult;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.currency.Money;
import ru.yandex.direct.intapi.entity.showconditions.service.validation.ModifyIntapiValidationResult;
import ru.yandex.direct.intapi.validation.model.IntapiError;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.pokazometer.PhraseResponse;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.DefectInfo;
import ru.yandex.direct.validation.result.PathNode;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.web.core.model.WebResponse;
import ru.yandex.direct.ytcomponents.statistics.model.StatValueAggregator;

import static java.lang.Math.max;
import static java.util.Arrays.asList;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.BooleanUtils.toIntegerObject;
import static ru.yandex.direct.core.entity.keyword.container.KeywordsModificationContainer.ADD_LIST_NAME;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.result.PathHelper.field;

public class ShowConditionsResponse implements WebResponse {

    public static final Set<Integer> PRESET_TRAFFIC_VOLUME_POSITIONS = ImmutableSet.of(5, 15, 75, 100);
    private static final double TRAFFIC_VOLUME_MULTIPLIER = 10_000.0;
    private static final Map<Integer, String> DEFAULT_EMPTY_POKAZOMETER_DATA =
            ImmutableMap.of(20, "0.00", 50, "0.00", 100, "0.00");

    private boolean isComplexOperationValid;

    private Map<Long, ResponseItem> items = new HashMap<>();

    // Ошибки верхнего уровня: результат валидации на уровне intapi.
    private List<String> errors = new ArrayList<>();

    private boolean isSuccessful = true;

    private final Currency clientCurrency;

    private ShowConditionsResponse(boolean isComplexOperationValid, Currency clientCurrency) {
        this.isComplexOperationValid = isComplexOperationValid;
        this.clientCurrency = clientCurrency;
    }

    public static ShowConditionsResponse failed(List<IntapiError> errors) {
        return new ShowConditionsResponse(false, null)
                .withErrors(mapList(errors, IntapiError::getText));
    }

    public static ShowConditionsResponse executedOperation(boolean isComplexOperationValid, Currency clientCurrency) {
        return new ShowConditionsResponse(isComplexOperationValid, clientCurrency);
    }

    public Map<Long, ResponseItem> getItems() {
        return items;
    }

    public ShowConditionsResponse withItems(Map<Long, ResponseItem> items) {
        this.items = items;
        return this;
    }

    public List<String> getErrors() {
        return errors;
    }

    public ShowConditionsResponse withErrors(List<String> errors) {
        this.errors = errors;
        return this;
    }

    @Override
    public boolean isSuccessful() {
        return isSuccessful;
    }

    @SuppressWarnings("checkstyle:parameternumber")
    public ShowConditionsResponse addKeywordResult(
            Result<KeywordsModificationResult> keywordsModificationResult,
            ModifyIntapiValidationResult intapiValidationResult,
            KeywordsModifyOperation keywordsModifyOperation,
            Map<Integer, String> inputKeywordPhrasesToAddMap,
            List<Pair<Long, ModelChanges<Keyword>>> keywordsToUpdate,
            List<Keyword> modifiedKeywords,
            Map<Long, StatValueAggregator> statDataByKeywordIds,
            List<KeywordBidPokazometerData> pokazometerData
    ) {
        if (keywordsModifyOperation == null) {
            return this;
        }

        Map<Integer, Pair<Long, ModelChanges<Keyword>>> keywordsToUpdateMap = EntryStream.of(keywordsToUpdate).toMap();

        Map<Long, KeywordBidPokazometerData> pokazometerDataMap =
                listToMap(pokazometerData, KeywordBidPokazometerData::getKeywordId);

        Map<Long, Keyword> modifiedKeywordsByIdMap = listToMap(modifiedKeywords, Keyword::getId);

        if (isComplexOperationValid && keywordsModificationResult != null) {
            addSuccessAddedKeywords(keywordsModifyOperation,
                    keywordsModificationResult.getResult().getAddResults(),
                    modifiedKeywordsByIdMap, statDataByKeywordIds, pokazometerDataMap,
                    keywordsModificationResult.getResult().getSourceAdGroupIdByCopiedAdGroupId());

            addSuccessUpdatedKeywords(keywordsModifyOperation,
                    keywordsModificationResult.getResult().getUpdateResults(),
                    keywordsToUpdateMap, modifiedKeywordsByIdMap, statDataByKeywordIds, pokazometerDataMap);

            addSuccessUpdatedOtherExistingKeywords(keywordsModifyOperation, keywordsToUpdateMap);
        }

        //в этом методе еще обрабатывается ситуация с превышением кол-ва фраз в группе при добавлении фразы
        addKeywordErrorMessages(keywordsModificationResult, intapiValidationResult,
                inputKeywordPhrasesToAddMap);
        return this;
    }

    public ShowConditionsResponse addRelevanceMatchResult(
            Result<RelevanceMatchModificationResult> relevanceMatchModificationResult,
            ModifyIntapiValidationResult relevanceMatchesIntapiValidationResult) {
        if (relevanceMatchModificationResult != null && !isComplexOperationValid) {
            addErrorMessages(relevanceMatchesIntapiValidationResult.getAddValidationInfo());
            addErrorMessages(relevanceMatchesIntapiValidationResult.getUpdateValidationInfo());
            addErrorMessages(relevanceMatchesIntapiValidationResult.getDeleteValidationInfo());
        }

        return this;
    }

    public ShowConditionsResponse addRetargetingResult(
            Map<Long, Long> retargetingIdToAdGroupId,
            MassResult<Long> resultEdited, MassResult<Long> resultDeleted,
            ModifyIntapiValidationResult retargetingsIntapiValidationResult
    ) {
        if (isComplexOperationValid) {
            addSuccessEditedRetargetingIds(resultEdited, retargetingIdToAdGroupId);
            addSuccessDeletedRetargetingIds(resultDeleted, retargetingIdToAdGroupId);
        }

        addErrorMessages(retargetingsIntapiValidationResult.getUpdateValidationInfo());
        addErrorMessages(retargetingsIntapiValidationResult.getDeleteValidationInfo());

        return this;
    }

    private void addSuccessEditedRetargetingIds(MassResult<Long> resultEdited,
                                                Map<Long, Long> retargetingIdToAdGroupId) {
        List<Long> successEditedRetargetingIds = getSuccessItems(resultEdited);

        for (Long id : successEditedRetargetingIds) {
            Long adGroupId = retargetingIdToAdGroupId.get(id);
            items.computeIfAbsent(adGroupId, x -> new ResponseItem())
                    .getRetargetings()
                    .getSuccessEdited()
                    .add(id);
        }
    }

    private void addSuccessDeletedRetargetingIds(MassResult<Long> resultDeleted,
                                                 Map<Long, Long> retargetingIdToAdGroupId) {
        List<Long> successDeletedRetargetingIds = getSuccessItems(resultDeleted);

        for (Long id : successDeletedRetargetingIds) {
            Long adGroupId = retargetingIdToAdGroupId.get(id);
            items.computeIfAbsent(adGroupId, x -> new ResponseItem())
                    .getRetargetings()
                    .getSuccessDeleted()
                    .add(id);
        }
    }

    public ShowConditionsResponse addSearchRetargetingResult(
            Map<Long, Long> searchRetargetingIdToAdGroupId,
            MassResult<Long> resultDeleted,
            ModifyIntapiValidationResult searchRetargetingsIntapiValidationResult
    ) {
        if (isComplexOperationValid) {
            addSuccessDeletedSearchRetargetingIds(resultDeleted, searchRetargetingIdToAdGroupId);
        }

        addErrorMessages(searchRetargetingsIntapiValidationResult.getDeleteValidationInfo());

        return this;
    }

    private void addSuccessDeletedSearchRetargetingIds(MassResult<Long> resultDeleted,
                                                       Map<Long, Long> searchRetargetingIdToAdGroupId) {
        List<Long> successDeletedRetargetingIds = getSuccessItems(resultDeleted);

        for (Long id : successDeletedRetargetingIds) {
            Long realId = BidModifierService.getRealId(id);
            Long adGroupId = searchRetargetingIdToAdGroupId.get(realId);
            items.computeIfAbsent(adGroupId, x -> new ResponseItem())
                    .getSearchRetargetings()
                    .getSuccessDeleted()
                    .add(realId);
        }
    }

    public ShowConditionsResponse addTargetInterestsResult(
            Map<Long, Long> targetInterestsIdToAdGroupId,
            MassResult<Long> resultEdited,
            MassResult<Long> resultDeleted,
            ModifyIntapiValidationResult intapiValidationResult
    ) {
        if (isComplexOperationValid) {
            addSuccessEditedTargetInterestsIds(resultEdited, targetInterestsIdToAdGroupId);
            addSuccessDeletedTargetInterestsIds(resultDeleted, targetInterestsIdToAdGroupId);
        }

        addErrorMessages(intapiValidationResult.getUpdateValidationInfo());
        addErrorMessages(intapiValidationResult.getDeleteValidationInfo());

        return this;
    }

    private void addSuccessEditedTargetInterestsIds(MassResult<Long> resultEdited,
                                                    Map<Long, Long> targetInterestsIdToAdGroupId) {
        List<Long> successEditedRetargetingIds = getSuccessItems(resultEdited);

        for (Long id : successEditedRetargetingIds) {
            Long adGroupId = targetInterestsIdToAdGroupId.get(id);
            items.computeIfAbsent(adGroupId, x -> new ResponseItem())
                    .getTargetInterests()
                    .getSuccessEdited()
                    .add(id);
        }
    }

    private void addSuccessDeletedTargetInterestsIds(MassResult<Long> resultDeleted,
                                                     Map<Long, Long> targetInterestsIdToAdGroupId) {
        List<Long> successDeletedTargetInterestsIds = getSuccessItems(resultDeleted);

        for (Long id : successDeletedTargetInterestsIds) {
            Long adGroupId = targetInterestsIdToAdGroupId.get(id);
            items.computeIfAbsent(adGroupId, x -> new ResponseItem())
                    .getTargetInterests()
                    .getSuccessDeleted()
                    .add(id);
        }
    }

    private void addSuccessAddedKeywords(KeywordsModifyOperation keywordsModifyOperation,
                                         List<AddedKeywordInfo> addedKeywordInfoList,
                                         Map<Long, Keyword> keywordsByIdMap,
                                         Map<Long, StatValueAggregator> statDataByKeywordIds,
                                         Map<Long, KeywordBidPokazometerData> pokazometerDataMap,
                                         Map<Long, Long> copiedAdGroupIdToSourceAdGroupId) {
        if (addedKeywordInfoList == null) {
            return;
        }

        Map<Integer, List<TrafaretBidItem>> trafaretBids = keywordsModifyOperation.getTrafaretBidsByAddIndexMap();

        for (int idx = 0; idx < addedKeywordInfoList.size(); idx++) {
            AddedKeywordInfo addedKeywordInfo = addedKeywordInfoList.get(idx);
            copiedAdGroupIdToSourceAdGroupId = nvl(copiedAdGroupIdToSourceAdGroupId, emptyMap());

            Long adGroupId = addedKeywordInfo.getAdGroupId();
            List<String> addedMinuses = addedKeywordInfo.getAddedMinuses();
            Long keywordId = addedKeywordInfo.getId();
            Keyword keyword = keywordsByIdMap.get(keywordId);

            ResponseItem responseItem = items.computeIfAbsent(adGroupId, x -> new ResponseItem());
            if (addedKeywordInfo.isAdded()) {
                Integer phrasesAddedQuantity = ObjectUtils.defaultIfNull(responseItem.getPhrasesAddedQuantity(), 0);
                responseItem
                        .withPhrasesAddedQuantity(++phrasesAddedQuantity);
            }

            KeywordItemResponse keywordItemResponse = responseItem.getKeywords()
                    .computeIfAbsent(keywordId, k -> new KeywordItemResponse());

            Long sourceAdGroupId = copiedAdGroupIdToSourceAdGroupId.get(adGroupId);
            if (sourceAdGroupId != null) {
                responseItem.withCopiedFromAdGroupId(sourceAdGroupId);
            }
            responseItem
                    .getKeywords()
                    .computeIfAbsent(addedKeywordInfo.getId(), k -> new KeywordItemResponse())
                    .withPhrase(addedKeywordInfo.getResultPhrase())
                    .withDuplicate(addedKeywordInfo.isAdded() ? 0 : 1)
                    .withPhraseUngluedSuffix(
                            addedMinuses != null ? addedMinuses.stream().collect(Collectors.joining(" -", " -", ""))
                                    : null);

            // дополнительная информация по торгам, статистика и ставки добавляются, если ключевик был сохранён
            if (addedKeywordInfo.isAdded()) {
                keywordItemResponse
                        .withPrice(toNullableString(keyword.getPrice()))
                        .withPriceContext(toNullableString(keyword.getPriceContext()))
                        .withPhraseId(toNullableString(keyword.getPhraseBsId()))
                        .withNormPhrase(keyword.getNormPhrase())
                        .withWordsCount(keyword.getWordsCount())
                        .withPriceForCoverage(fromKeywordBidPokazometerData(pokazometerDataMap.get(keywordId)))
                        .withTrafficVolume(fromTrafaretBidItem(trafaretBids.get(idx), clientCurrency));

                fillStatisticDataToKeywordResponse(statDataByKeywordIds, keywordItemResponse, keywordId);
            }
        }
    }

    private void addSuccessUpdatedKeywords(KeywordsModifyOperation keywordsModifyOperation,
                                           List<UpdatedKeywordInfo> updatedKeywordInfoList,
                                           Map<Integer, Pair<Long, ModelChanges<Keyword>>> keywordsToUpdateMap,
                                           Map<Long, Keyword> keywordsByIdMap,
                                           Map<Long, StatValueAggregator> statDataByKeywordIds,
                                           Map<Long, KeywordBidPokazometerData> pokazometerDataMap) {
        if (updatedKeywordInfoList == null) {
            return;
        }

        Map<Integer, List<TrafaretBidItem>> trafaretBids = keywordsModifyOperation.getTrafaretBidsByUpdateIndexMap();

        for (int idx = 0; idx < updatedKeywordInfoList.size(); idx++) {
            UpdatedKeywordInfo updatedKeywordInfo = updatedKeywordInfoList.get(idx);
            Pair<Long, ModelChanges<Keyword>> inputKeyword = keywordsToUpdateMap.get(idx);
            List<String> addedMinuses = updatedKeywordInfo.getAddedMinuses();
            List<StopwordsFixation> fixations = updatedKeywordInfo.getFixations();
            Long keywordId = updatedKeywordInfo.getId();

            KeywordItemResponse keywordItemResponse =
                    items.computeIfAbsent(inputKeyword.getLeft(), x -> new ResponseItem())
                            .getKeywords()
                            .computeIfAbsent(inputKeyword.getRight().getId(), k -> new KeywordItemResponse());

            keywordItemResponse
                    .withId(updatedKeywordInfo.getId())
                    .withPhrase(updatedKeywordInfo.getPhraseBeforeUnglue())
                    .withDuplicate(updatedKeywordInfo.isDeleted() ? 1 : 0)
                    .withIsSuspended(toIntegerObject(updatedKeywordInfo.getIsSuspended()))
                    .withFixations(fixations != null ? mapList(fixations,
                            fixation -> asList(fixation.getDestSubstring(), fixation.getSourceSubstring())) : null)
                    .withPhraseUngluedSuffix(
                            addedMinuses != null ? addedMinuses.stream().collect(Collectors.joining(" -", " -", ""))
                                    : null);

            // дополнительная информация по торгам, статистика и ставки добавляются, если ключевик не был задублирован
            if (!updatedKeywordInfo.isDeleted() && keywordsByIdMap.containsKey(keywordId)) {
                Keyword keyword = keywordsByIdMap.get(keywordId);

                keywordItemResponse
                        .withPrice(toNullableString(keyword.getPrice()))
                        .withPriceContext(toNullableString(keyword.getPriceContext()))
                        .withPhraseId(toNullableString(keyword.getPhraseBsId()))
                        .withNormPhrase(keyword.getNormPhrase())
                        .withWordsCount(keyword.getWordsCount())
                        .withPriceForCoverage(fromKeywordBidPokazometerData(pokazometerDataMap.get(keywordId)))
                        .withTrafficVolume(fromTrafaretBidItem(trafaretBids.get(idx), clientCurrency));

                fillStatisticDataToKeywordResponse(statDataByKeywordIds, keywordItemResponse, keywordId);
            }
        }
    }

    /**
     * Привести объект к строке.
     * Если передан {@code null}, вернёт {@code null}.
     */
    @Nullable
    private static String toNullableString(Object obj) {
        return Objects.toString(obj, null);
    }

    private static Map<Integer, String> fromKeywordBidPokazometerData(KeywordBidPokazometerData data) {
        if (data == null) {
            // todo DIRECT-81272 дефолтный ответ если значений показометр не вернул, заглушка для того, чтобы не
            // рефакторить фронт.
            return DEFAULT_EMPTY_POKAZOMETER_DATA;
        }

        return EntryStream.of(data.getCoverageWithPrices())
                .mapKeys(PhraseResponse.Coverage::getPercentage)
                .mapValues(Money::bigDecimalValue)
                .mapValues(ShowConditionsResponse::toNullableString)
                .toMap();
    }


    private static Map<Integer, TrafficVolumeItem> fromTrafaretBidItem(@Nullable List<TrafaretBidItem> items,
                                                                       Currency clientCurrency) {
        if (items == null) {
            return null;
        }

        int maxPosition = 0;
        Map<Integer, TrafficVolumeItem> trafficVolumeItems = new LinkedHashMap<>();
        for (TrafaretBidItem item : items) {
            int position = (int) Math.round(item.getPositionCtrCorrection() / TRAFFIC_VOLUME_MULTIPLIER);
            maxPosition = max(position, maxPosition);

            boolean dontShowBidPrice = item.getBid().bigDecimalValue().compareTo(clientCurrency.getMaxShowBid()) > 0;
            boolean isShowInTable = PRESET_TRAFFIC_VOLUME_POSITIONS.contains(position);

            TrafficVolumeItem trafficVolumeItem = new TrafficVolumeItem()
                    .withAmnestyPrice(item.getPrice().micros())
                    .withBidPrice(item.getBid().micros())
                    .withDontShowBidPrice(dontShowBidPrice ? 1 : null)
                    .withShowInTable(isShowInTable ? 1 : null);

            trafficVolumeItems.put(position, trafficVolumeItem);
        }

        if (trafficVolumeItems.containsKey(maxPosition)) {
            trafficVolumeItems.get(maxPosition).withShowInTable(1);
        }

        return trafficVolumeItems;
    }


    /**
     * Заполнение данными из статистики, идентификатор ключевика должен быть уже заполнен в ответе.
     */
    private void fillStatisticDataToKeywordResponse(Map<Long, StatValueAggregator> statDataByKeywordIds,
                                                    KeywordItemResponse keywordItemResponse, long keywordId) {
        long shows = 0L;
        long clicks = 0L;
        BigDecimal ctr = BigDecimal.ZERO;

        StatValueAggregator statItem = statDataByKeywordIds.get(keywordId);
        if (statItem != null) {
            clicks = statItem.getSearchClicks();
            shows = statItem.getSearchShows();
            if (shows != 0L) {
                ctr = BigDecimal.valueOf(((double) clicks) / shows)
                        .movePointRight(2); // приведение к значению в процентах
            }
        }

        keywordItemResponse
                .withCtr(ctr.setScale(2, BigDecimal.ROUND_HALF_UP).toString()) // принудительно 2 знака после запятойх
                .withShows(shows)
                .withClicks(clicks);
    }

    private void addSuccessUpdatedOtherExistingKeywords(
            KeywordsModifyOperation keywordsModifyOperation,
            Map<Integer, Pair<Long, ModelChanges<Keyword>>> keywordsToUpdateMap
    ) {
        if (keywordsModifyOperation == null || keywordsToUpdateMap.isEmpty()) {
            return;
        }

        keywordsModifyOperation.getAffectedKeywordInfoListByUpdate().forEach(affectedKeywordInfo -> {
            List<String> addedMinuses = affectedKeywordInfo.getAddedMinuses();
            items.computeIfAbsent(affectedKeywordInfo.getAdGroupId(), x -> new ResponseItem())
                    .getKeywords()
                    .computeIfAbsent(affectedKeywordInfo.getId(), k -> new KeywordItemResponse())
                    .withId(affectedKeywordInfo.getId())
                    .withPhrase(affectedKeywordInfo.getSourcePhrase())
                    .withDuplicate(0)
                    .withIsSuspended(toIntegerObject(affectedKeywordInfo.getIsSuspended()))
                    .withPhraseUngluedSuffix(
                            addedMinuses != null ? addedMinuses.stream().collect(Collectors.joining(" -", " -", ""))
                                    : null);
        });
    }

    private void addKeywordErrorMessages(Result<KeywordsModificationResult> result,
                                         ModifyIntapiValidationResult intapiValidationResult,
                                         Map<Integer, String> inputKeywordPhrasesToAddMap) {
        if (result == null || result.getValidationResult() == null) {
            return;
        }
        ValidationResult<?, Defect> vrAdd = result.getValidationResult()
                .getSubResults().get(field(ADD_LIST_NAME));
        addKeywordAddErrorMessages(vrAdd, intapiValidationResult.getAddValidationInfo(), inputKeywordPhrasesToAddMap);

        addKeywordCopyAdGroupsErrorMessages(intapiValidationResult.getCopyAdGroupsError());

        addErrorMessages(intapiValidationResult.getUpdateValidationInfo());
        addErrorMessages(intapiValidationResult.getDeleteValidationInfo());
    }

    private void addKeywordCopyAdGroupsErrorMessages(List<IntapiError> validationErrors) {
        if (validationErrors == null) {
            return;
        }

        isSuccessful = validationErrors.isEmpty();
        errors.addAll(mapList(validationErrors, IntapiError::getText));
    }

    private void addKeywordAddErrorMessages(
            ValidationResult<?, Defect> validationResult,
            Map<Integer, ModifyIntapiValidationResult.ValidationContainer> validationContainerMap,
            Map<Integer, String> inputKeywordPhrasesToAddMap
    ) {
        if (validationResult == null) {
            return;
        }
        validationResult.getSubResults().forEach((pathNode, subResult) -> {
            int idx = ((PathNode.Index) pathNode).getIndex();
            ModifyIntapiValidationResult.ValidationContainer container = validationContainerMap.get(idx);

            List<Defect> defects = mapList(subResult.flattenErrors(), DefectInfo::getDefect);
            Defect<KeywordsLimitDefectParams> maxKeywordsExceeded = getPhraseLimitExceededError(defects);
            Long adGroupId = container.getAdGroupId();
            String phrase = inputKeywordPhrasesToAddMap.get(idx);
            ResponseItem responseItem = items.computeIfAbsent(adGroupId, x -> new ResponseItem());

            if (maxKeywordsExceeded != null) {
                responseItem
                        .withIsGroupOversized(1)
                        .withPhrasesExceedsLimitQuantity(
                                maxKeywordsExceeded.params().getNewKeywordsExceededCount().intValue());
            }

            //игнорируем ошибку превышения кол-ва фраз
            List<IntapiError> intapiErrors = container.getErrors().stream()
                    .filter(t -> !KeywordDefectIds.Keyword.MAX_KEYWORDS_PER_AD_GROUP_EXCEEDED.getCode()
                            .equals(t.getCode()))
                    .collect(toList());
            if (!intapiErrors.isEmpty()) {
                if (responseItem.getErrors() == null) {
                    responseItem.withErrors(new HashSet<>());
                }
                if (responseItem.getErrorsByPhrases() == null) {
                    responseItem.withErrorsByPhrases(new ArrayList<>());
                }
                KeywordAddItemError itemError = responseItem.getErrorsByPhrases().stream()
                        .filter(item -> item.getPhrase().equals(phrase))
                        .findFirst()
                        .orElse(null);
                if (itemError == null) {
                    itemError = new KeywordAddItemError().withPhrase(phrase);
                    responseItem.getErrorsByPhrases().add(itemError);
                }
                itemError.getErrors().addAll(listToSet(intapiErrors, IntapiError::getText));
            }
        });

        if (validationResult.hasAnyErrors()) {
            isSuccessful = false;
        }
    }

    @SuppressWarnings("unchecked")
    private Defect<KeywordsLimitDefectParams> getPhraseLimitExceededError(
            List<Defect> defects) {
        return defects.stream()
                .filter(t -> KeywordDefectIds.Keyword.MAX_KEYWORDS_PER_AD_GROUP_EXCEEDED.equals(t.defectId()))
                .findFirst()
                .orElse(null);
    }

    private void addErrorMessages(Map<Integer, ModifyIntapiValidationResult.ValidationContainer> validationInfoMap) {
        if (validationInfoMap == null) {
            return;
        }

        isSuccessful = validationInfoMap.values().stream()
                .allMatch(vc -> vc.getErrors().isEmpty());

        validationInfoMap.forEach((index, info) -> {
            if (!info.getErrors().isEmpty()) {
                Long adGroupId = info.getAdGroupId();

                ResponseItem responseItem = items.computeIfAbsent(adGroupId, x -> new ResponseItem());
                if (responseItem.getErrors() == null) {
                    responseItem.withErrors(new HashSet<>());
                }
                responseItem.getErrors().addAll(mapList(info.getErrors(), IntapiError::getText));
            }
        });
    }

    private List<Long> getSuccessItems(MassResult<Long> result) {
        if (result != null) {
            return result.getResult().stream()
                    .filter(Result::isSuccessful)
                    .map(Result::getResult)
                    .collect(toList());
        } else {
            return emptyList();
        }
    }

    public ShowConditionsResponse addBannersStatusesResult(List<BannerItemResponse> bannerItemsResponse) {
        for (BannerItemResponse bannerItemResponse : bannerItemsResponse) {
            Long adgroupId = bannerItemResponse.getAdgroupId();
            Long bannerId = bannerItemResponse.getBannerId();

            bannerItemResponse.removeBannerAndAdgroupIds();

            items.computeIfAbsent(adgroupId, x -> new ResponseItem())
                    .getBanners()
                    .putIfAbsent(bannerId, bannerItemResponse);
        }
        return this;
    }
}
