package ru.yandex.direct.intapi.entity.showconditions.service;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.TranslationService;
import ru.yandex.direct.core.entity.banner.type.moderation.BannerStatusRepository;
import ru.yandex.direct.core.entity.banner.type.moderation.model.BannerStatusInfo;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.i18n.Translatable;
import ru.yandex.direct.i18n.types.ConcatTranslatable;
import ru.yandex.direct.intapi.entity.showconditions.BannerStatusTranslations;
import ru.yandex.direct.intapi.entity.showconditions.model.response.BannerItemResponse;

import static ru.yandex.direct.utils.FunctionalUtils.mapList;


@Service
@ParametersAreNonnullByDefault
public class BannerStatusService {

    private final BannerStatusRepository bannerStatusRepository;
    private final ShardHelper shardHelper;
    private final TranslationService translationService;

    @Autowired
    public BannerStatusService(BannerStatusRepository bannerStatusRepository,
                               ShardHelper shardHelper,
                               TranslationService translationService) {
        this.bannerStatusRepository = bannerStatusRepository;
        this.shardHelper = shardHelper;
        this.translationService = translationService;
    }

    private static BannerStatusTranslations bannerTranslations() {
        return BannerStatusTranslations.INSTANCE;
    }

    public List<BannerItemResponse> getBannersResponse(ClientId clientId, Collection<Long> bannerIds) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        List<BannerStatusInfo> bannersStatusInfo = bannerStatusRepository.getBannersStatusInfo(shard, bannerIds);
        return mapList(bannersStatusInfo, bannerStatusInfo ->
                new BannerItemResponse(bannerStatusInfo.getBannerId(), bannerStatusInfo.getAdgroupId())
                        .withStatusActive(bannerStatusInfo.isStatusActive())
                        .withStatusShow(bannerStatusInfo.isStatusShow())
                        .withStatusModerate(bannerStatusInfo.getStatusModerate())
                        .withBannerID(bannerStatusInfo.getBsBannerId())
                        .withTextStatus(calcTextStatus(bannerStatusInfo)));
    }

    /*
        По флажкам определяем какой текстовый вариант статуса должен быть у объявления.
     */
    private String calcTextStatus(BannerStatusInfo statusInfo) {

        List<Translatable> textStatus = new ArrayList<>();
        if (statusInfo.isShowAccepted()
                && statusInfo.isModerateWait()
                && !statusInfo.isModerateContactDeclined()
                && !statusInfo.isModerateDeclined()) {
            textStatus.add(bannerTranslations().getToShow());
        }
        if (statusInfo.isShowActive()) {
            textStatus.add(bannerTranslations().showsAreGoing());
        }
        if (!statusInfo.isStopped() && statusInfo.isModerateBannerAccepted()) {
            textStatus.add(bannerTranslations().adAccepted());
        }
        if (statusInfo.isModerateContactWait()) {
            textStatus.add(bannerTranslations().contactInfoWaitModeration());
        }
        if (statusInfo.isModerateSitelinksWait()) {
            textStatus.add(bannerTranslations().sitelinksWaitModeration());
        }
        if (statusInfo.isModerateImageWait()) {
            textStatus.add(bannerTranslations().imageWaitModeration());
        }
        if (statusInfo.isModerateDisplayHrefWait() && !statusInfo.isDraft()) {
            textStatus.add(bannerTranslations().displayHrefWaitModeration());
        }
        if (statusInfo.isModerateVideoAdditionWait()) {
            textStatus.add(bannerTranslations().videoAdditionsWaitModeration());
        }
        if (statusInfo.isModerateTurbolandingWait()) {
            textStatus.add(bannerTranslations().turboLandingWaitModeration());
        }
        if (statusInfo.isModerateWait() && !statusInfo.isShowAccepted()) {
            textStatus.add(bannerTranslations().waitModeration());
        }
        if (statusInfo.isDraft()) {
            textStatus.add(bannerTranslations().draft());
        }
        if (statusInfo.isModerateDeclined() && !statusInfo.isModerateWait()) {
            textStatus.add(bannerTranslations().declineModeration());
        }

        // Если отклонено больше, чем по одному пункту, то составляем отдельны текст с перечислением причин отклонения
        long numFlags = StreamEx.of(statusInfo.isModerateContactDeclined(),
                statusInfo.isModeratePhrasesDeclined(),
                statusInfo.isModerateSitelinksDeclined(),
                statusInfo.isModerateImageDeclined(),
                statusInfo.isModerateCalloutDeclined(),
                statusInfo.isModerateDisplayHrefDeclined(),
                statusInfo.isModerateVideoAdditionDeclined(),
                statusInfo.isModerateTurbolandingDeclined())
                .filter(isTrue -> isTrue)
                .count();

        if (numFlags > 1) {
            List<Translatable> textMultiplyDecline = new ArrayList<>();
            if (statusInfo.isModerateContactDeclined()) {
                textMultiplyDecline.add(bannerTranslations().contactInfoMultiplyDeclined());
            }
            if (statusInfo.isModeratePhrasesDeclined()) {
                textMultiplyDecline.add(bannerTranslations().keywordsMultiplyDeclined());
            }
            if (statusInfo.isModerateSitelinksDeclined()) {
                textMultiplyDecline.add(bannerTranslations().sitelinksMultiplyDeclined());
            }
            if (statusInfo.isModerateImageDeclined()) {
                textMultiplyDecline.add(bannerTranslations().imageMultiplyDeclined());
            }
            if (statusInfo.isModerateCalloutDeclined()) {
                textMultiplyDecline.add(bannerTranslations().calloutMultiplyDeclined());
            }
            if (statusInfo.isModerateDisplayHrefDeclined()) {
                textMultiplyDecline.add(bannerTranslations().displayHrefMultiplyDeclined());
            }
            if (statusInfo.isModerateVideoAdditionDeclined()) {
                textMultiplyDecline.add(bannerTranslations().videoAdditionsMultiplyDeclined());
            }
            if (statusInfo.isModerateTurbolandingDeclined()) {
                textMultiplyDecline.add(bannerTranslations().turboLandingMultiplyDeclined());
            }

            textStatus.add(
                    new ConcatTranslatable(bannerTranslations().multiplyDeclined(),
                            new ConcatTranslatable(", ", textMultiplyDecline)));
        } else {
            // Иначе только одну причину пишем.
            if (statusInfo.isModerateContactDeclined()) {
                textStatus.add(bannerTranslations().contactInfoDeclinedModeration());
            } else if (statusInfo.isModeratePhrasesDeclined()) {
                textStatus.add(bannerTranslations().keywordsDeclinedModeration());
            } else if (statusInfo.isModerateSitelinksDeclined()) {
                textStatus.add(bannerTranslations().sitelinksDeclinedModeration());
            } else if (statusInfo.isModerateImageDeclined()) {
                textStatus.add(bannerTranslations().imageDeclinedModeration());
            } else if (statusInfo.isModerateCalloutDeclined()) {
                textStatus.add(bannerTranslations().calloutDeclinedModeration());
            } else if (statusInfo.isModerateDisplayHrefDeclined()) {
                textStatus.add(bannerTranslations().displayHrefDeclinedModeration());
            } else if (statusInfo.isModerateVideoAdditionDeclined()) {
                textStatus.add(bannerTranslations().videoAdditionsDeclinedModeration());
            } else if (statusInfo.isModerateTurbolandingDeclined()) {
                textStatus.add(bannerTranslations().turboLandingDeclinedModeration());
            }
        }

        if (statusInfo.isActivation() && !statusInfo.isModerateWait()) {
            textStatus.add(bannerTranslations().activating());
        }
        if (statusInfo.isBlNothingGenerated()) {
            textStatus.add(bannerTranslations().blNothingGenerated());
        } else if (statusInfo.isBlProcessing()) {
            textStatus.add(bannerTranslations().blProcession());
        }
        String statuses = String.join(" ", mapList(textStatus, translationService::translate)).trim();
        return StringUtils.stripEnd(statuses, ".");
    }
}

