package ru.yandex.direct.intapi.entity.showconditions.service;

import java.time.Period;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import one.util.streamex.EntryStream;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.model.AdGroupSimple;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.service.AdGroupService;
import ru.yandex.direct.core.entity.banner.repository.BannerRelationsRepository;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierType;
import ru.yandex.direct.core.entity.bidmodifiers.service.BidModifierDeleteOperation;
import ru.yandex.direct.core.entity.bidmodifiers.service.BidModifierService;
import ru.yandex.direct.core.entity.bids.container.KeywordBidPokazometerData;
import ru.yandex.direct.core.entity.bids.service.KeywordBidDynamicDataService;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.keyword.container.AddedKeywordInfo;
import ru.yandex.direct.core.entity.keyword.container.KeywordsModificationContainer;
import ru.yandex.direct.core.entity.keyword.container.KeywordsModificationResult;
import ru.yandex.direct.core.entity.keyword.container.UpdatedKeywordInfo;
import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.core.entity.keyword.service.KeywordModifyOperationFactory;
import ru.yandex.direct.core.entity.keyword.service.KeywordService;
import ru.yandex.direct.core.entity.keyword.service.KeywordsModifyOperation;
import ru.yandex.direct.core.entity.relevancematch.container.RelevanceMatchModification;
import ru.yandex.direct.core.entity.relevancematch.container.RelevanceMatchModificationResult;
import ru.yandex.direct.core.entity.relevancematch.model.RelevanceMatch;
import ru.yandex.direct.core.entity.relevancematch.service.RelevanceMatchModifyOperation;
import ru.yandex.direct.core.entity.relevancematch.service.RelevanceMatchService;
import ru.yandex.direct.core.entity.retargeting.model.Retargeting;
import ru.yandex.direct.core.entity.retargeting.service.RetargetingDeleteOperation;
import ru.yandex.direct.core.entity.retargeting.service.RetargetingService;
import ru.yandex.direct.core.entity.retargeting.service.RetargetingUpdateOperation;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.intapi.entity.adqualityexport.AdQualityExportLogger;
import ru.yandex.direct.intapi.entity.showconditions.model.request.ShowConditionsRequest;
import ru.yandex.direct.intapi.entity.showconditions.model.response.BannerItemResponse;
import ru.yandex.direct.intapi.entity.showconditions.model.response.ShowConditionsResponse;
import ru.yandex.direct.intapi.entity.showconditions.service.validation.ModifyIntapiValidationResult;
import ru.yandex.direct.intapi.entity.showconditions.service.validation.ShowConditionsValidationResultConverter;
import ru.yandex.direct.intapi.entity.showconditions.service.validation.ShowConditionsValidationService;
import ru.yandex.direct.intapi.validation.kernel.ValidationResultConversionService;
import ru.yandex.direct.intapi.validation.model.IntapiError;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.utils.FunctionalUtils;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.web.core.model.WebResponse;
import ru.yandex.direct.ytcomponents.statistics.model.DateRange;
import ru.yandex.direct.ytcomponents.statistics.model.PhraseStatisticsRequest;
import ru.yandex.direct.ytcomponents.statistics.model.PhraseStatisticsResponse;
import ru.yandex.direct.ytcomponents.statistics.model.StatValueAggregator;
import ru.yandex.direct.ytcore.entity.statistics.service.RecentStatisticsService;

import static java.util.Collections.emptyList;
import static org.springframework.util.ObjectUtils.isEmpty;
import static ru.yandex.direct.intapi.entity.statistic.model.PhraseStatisticConverter.getDateRangeFromPeriod;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.ytcore.entity.statistics.service.RecentStatisticsService.DEFAULT_STATISTIC_PERIOD;

@Service
public class ShowConditionsService {

    private static final Logger logger = LoggerFactory.getLogger(ShowConditionsService.class);

    private final ValidationResultConversionService validationResultConversionService;
    private final RbacService rbacService;
    private final RelevanceMatchService relevanceMatchService;
    private final RetargetingService retargetingService;
    private final RecentStatisticsService recentStatisticsService;
    private final ClientService clientService;
    private final KeywordService keywordService;
    private final KeywordModifyOperationFactory keywordModifyOperationFactory;
    private final BannerStatusService bannerStatusService;
    private final KeywordBidDynamicDataService keywordBidDynamicDataService;
    private final AdGroupService adGroupService;
    private final BannerRelationsRepository bannerRelationsRepository;
    private final AdQualityExportLogger adQualityExportLogger;
    private final BidModifierService bidModifierService;


    // Используем RetargetingService для Таргетов по интересам так как в ядре, перле и БД они реализованы
    // через одни и те же объекты. В данный момент RetargetingService полностью удовлетворяет потребности сервиса
    // таргетов по интересам. Если их поведение будет отличаться - появится необходимость их разнести.
    private final RetargetingService targetInterestsService;

    private final ShowConditionsValidationService validationService;
    private final ShowConditionsValidationResultConverter showConditionsValidationResultConverter;

    @Autowired
    @SuppressWarnings("checkstyle:parameternumber")
    public ShowConditionsService(
            ValidationResultConversionService validationResultConversionService,
            RbacService rbacService,
            RelevanceMatchService relevanceMatchService,
            RetargetingService retargetingService,
            RecentStatisticsService recentStatisticsService,
            ClientService clientService,
            KeywordService keywordService,
            KeywordModifyOperationFactory keywordModifyOperationFactory,
            KeywordBidDynamicDataService keywordBidDynamicDataService,
            AdGroupService adGroupService,
            BannerRelationsRepository bannerRelationsRepository,
            RetargetingService targetInterestsService,
            BannerStatusService bannerStatusService,
            ShowConditionsValidationService validationService,
            ShowConditionsValidationResultConverter showConditionsValidationResultConverter,
            AdQualityExportLogger adQualityExportLogger,
            BidModifierService bidModifierService) {
        this.validationResultConversionService = validationResultConversionService;
        this.rbacService = rbacService;
        this.relevanceMatchService = relevanceMatchService;
        this.keywordService = keywordService;
        this.keywordModifyOperationFactory = keywordModifyOperationFactory;
        this.retargetingService = retargetingService;
        this.clientService = clientService;
        this.recentStatisticsService = recentStatisticsService;
        this.keywordBidDynamicDataService = keywordBidDynamicDataService;
        this.adGroupService = adGroupService;
        this.bannerRelationsRepository = bannerRelationsRepository;
        this.targetInterestsService = targetInterestsService;
        this.bannerStatusService = bannerStatusService;
        this.validationService = validationService;
        this.showConditionsValidationResultConverter = showConditionsValidationResultConverter;
        this.adQualityExportLogger = adQualityExportLogger;
        this.bidModifierService = bidModifierService;
    }

    @SuppressWarnings("checkstyle:methodlength")
    public WebResponse doOperations(long operatorUid, ClientId clientId, ShowConditionsRequest request) {
        ValidationResult<ShowConditionsRequest, Defect> vr = validationService.validate(request);

        if (vr.hasAnyErrors()) {
            List<IntapiError> errors = validationResultConversionService.buildValidationResponse(vr)
                    .validationResult()
                    .getErrors();

            return ShowConditionsResponse.failed(errors);
        }
        long clientUid = rbacService.getChiefByClientId(clientId);
        Currency clientCurrency = clientService.getWorkCurrency(clientId);

        // keywords

        List<Pair<String, Keyword>> pairsKeywordsToAdd = request.getKeywordsToAdd();
        List<Keyword> keywordsToAdd = mapList(pairsKeywordsToAdd, Pair::getValue);

        Map<Integer, String> inputKeywordPhrasesToAddMap =
                EntryStream.of(mapList(keywordsToAdd, Keyword::getPhrase)).toMap();
        // Список пар: adGroupId -> ModelChanges<Keyword>
        List<Pair<Long, ModelChanges<Keyword>>> keywordsToUpdate = request.getKeywordChangesToEdit();
        List<Pair<Long, Long>> keywordsToDelete = request.getAdGroupIdToKeywordToDelete();

        KeywordsModificationContainer keywordsModificationContainer;
        if (request.getCopyOversizeGroupFlag()) {
            keywordsModificationContainer = KeywordsModificationContainer
                    .addWithCopyOnOversize(keywordsToAdd);
        } else {
            keywordsModificationContainer = KeywordsModificationContainer
                    .addUpdateDelete(keywordsToAdd, mapList(keywordsToUpdate, Pair::getRight),
                            mapList(keywordsToDelete, Pair::getRight));
        }

        KeywordsModifyOperation keywordsModifyOperation = null;
        Result<KeywordsModificationResult> keywordsModificationResult = null;
        if (!keywordsModificationContainer.isEmpty()) {
            keywordsModifyOperation = keywordModifyOperationFactory.createKeywordsModifyOperation(
                    keywordsModificationContainer, operatorUid, clientId, clientUid);
            keywordsModificationResult = keywordsModifyOperation.prepare().orElse(null);
        }

        // relevance matches

        List<Pair<Long, ModelChanges<RelevanceMatch>>> relevanceMatchesToUpdate =
                request.getRelevanceMatchChangesToEdit();
        List<Pair<Long, Long>> relevanceMatchesToDelete = request.getAdGroupIdToRelevanceMatchToDelete();

        RelevanceMatchModification relevanceMatchModification = new RelevanceMatchModification()
                .withRelevanceMatchAdd(request.getRelevanceMatchesToAdd())
                .withRelevanceMatchUpdate(mapList(relevanceMatchesToUpdate, Pair::getRight))
                .withRelevanceMatchIdsDelete(mapList(relevanceMatchesToDelete, Pair::getRight));

        boolean hasRelevanceMatchModification = !relevanceMatchModification.getRelevanceMatchAdd().isEmpty() ||
                !relevanceMatchModification.getRelevanceMatchUpdate().isEmpty()
                || !relevanceMatchModification.getRelevanceMatchIdsDelete().isEmpty();

        RelevanceMatchModifyOperation relevanceMatchModifyOperation = null;
        Result<RelevanceMatchModificationResult> relevanceMatchModificationResult = null;
        if (hasRelevanceMatchModification) {
            relevanceMatchModifyOperation = relevanceMatchService
                    .createFullModifyOperation(clientId, clientUid, operatorUid, relevanceMatchModification, false,
                            null);
            relevanceMatchModificationResult = relevanceMatchModifyOperation.prepare().orElse(null);
        }

        // retargetings

        List<ModelChanges<Retargeting>> retargetingModelChanges = request.getRetargetingChangesToEdit();
        List<Long> retargetingIdsToDelete = request.getRetargetingIdsToDelete();
        Map<Long, Long> retargetingIdToAdGroupId = request.retargetingIdToAdGroupIdMap();

        RetargetingUpdateOperation retargetingUpdateOperation = null;
        MassResult<Long> retargetingsUpdateResult = null;
        if (!retargetingModelChanges.isEmpty()) {
            retargetingUpdateOperation = retargetingService.createUpdateOperation(Applicability.FULL,
                    retargetingModelChanges, operatorUid, clientId, clientUid);
            retargetingsUpdateResult = retargetingUpdateOperation.prepare().orElse(null);
        }

        RetargetingDeleteOperation retargetingDeleteOperation = null;
        MassResult<Long> retargetingsDeleteResult = null;
        if (!retargetingIdsToDelete.isEmpty()) {
            retargetingDeleteOperation = retargetingService
                    .createDeleteOperation(Applicability.FULL,
                            retargetingIdsToDelete, operatorUid, clientId, clientUid);
            retargetingsDeleteResult = retargetingDeleteOperation.prepare().orElse(null);
        }

        // search retargetings
        List<Long> searchRetargetingIdsToDelete = request.getSearchRetargetingIdsToDelete(); //сейчас только удаление
        Map<Long, Long> searchRetargetingIdToAdGroupId = request.searchRetargetingsToAdGroupIdMap();

        BidModifierDeleteOperation bidModifierDeleteOperation = null;
        MassResult<Long> searchRetargetingDeleteResult = null;
        if (!searchRetargetingIdsToDelete.isEmpty()) {
            List<Long> externalModifierIds = mapList(searchRetargetingIdsToDelete,
                    id -> BidModifierService.getExternalId(id, BidModifierType.RETARGETING_FILTER));
            bidModifierDeleteOperation = bidModifierService
                    .createDeleteOperation(externalModifierIds, clientId, operatorUid);
            searchRetargetingDeleteResult = bidModifierDeleteOperation.prepare().orElse(null);
        }

        // target interests

        List<ModelChanges<Retargeting>> targetInterestsModelChanges = request.getTargetInterestsChangesToEdit();
        List<Long> targetInterestsIdsToDelete = request.getTargetInterestsIdsToDelete();
        Map<Long, Long> targetInterestsIdToAdGroupId = request.targetInterestsIdToAdGroupIdMap();

        RetargetingUpdateOperation targetInterestsUpdateOperation = null;
        MassResult<Long> targetInterestsUpdateResult = null;
        if (!targetInterestsModelChanges.isEmpty()) {
            targetInterestsUpdateOperation = targetInterestsService.createUpdateOperation(Applicability.FULL,
                    targetInterestsModelChanges, operatorUid, clientId, clientUid);
            targetInterestsUpdateResult = targetInterestsUpdateOperation.prepare().orElse(null);
        }

        RetargetingDeleteOperation targetInterestsDeleteOperation = null;
        MassResult<Long> targetInterestsDeleteResult = null;
        if (!targetInterestsIdsToDelete.isEmpty()) {
            targetInterestsDeleteOperation = targetInterestsService
                    .createDeleteOperation(Applicability.FULL,
                            targetInterestsIdsToDelete, operatorUid, clientId, clientUid);
            targetInterestsDeleteResult = targetInterestsDeleteOperation.prepare().orElse(null);
        }

        List<Keyword> modifiedKeywords = null;
        Map<Long, StatValueAggregator> statDataByKeywordIds = null;
        List<KeywordBidPokazometerData> pokazometerData = null;

        boolean isComplexOperationValid = keywordsModificationResult == null
                && relevanceMatchModificationResult == null
                && retargetingsUpdateResult == null && retargetingsDeleteResult == null
                && targetInterestsUpdateResult == null && targetInterestsDeleteResult == null
                && searchRetargetingDeleteResult == null;
        if (isComplexOperationValid) {
            if (keywordsModifyOperation != null) {
                keywordsModificationResult = keywordsModifyOperation.apply();
            }
            if (relevanceMatchModifyOperation != null) {
                relevanceMatchModificationResult = relevanceMatchModifyOperation.apply();
            }
            if (retargetingUpdateOperation != null) {
                retargetingsUpdateResult = retargetingUpdateOperation.apply();
            }
            if (retargetingDeleteOperation != null) {
                retargetingsDeleteResult = retargetingDeleteOperation.apply();
            }
            if (bidModifierDeleteOperation != null) {
                searchRetargetingDeleteResult = bidModifierDeleteOperation.apply();
            }
            if (targetInterestsUpdateOperation != null) {
                targetInterestsUpdateResult = targetInterestsUpdateOperation.apply();
            }
            if (targetInterestsDeleteOperation != null) {
                targetInterestsDeleteResult = targetInterestsDeleteOperation.apply();
            }

            if (keywordsModificationResult != null) {
                modifiedKeywords = getModifiedKeywords(clientId, keywordsModificationResult);
                Map<Long, AdGroupSimple> simpleAdGroups =
                        adGroupService.getSimpleAdGroups(clientId, mapList(modifiedKeywords, Keyword::getAdGroupId));

                // фильтруем ключевики, чтобы не ходить за статистикой и в показометр для тех групп, где это не нужно.
                List<Keyword> keywordsForRequest = filterList(modifiedKeywords,
                        k -> simpleAdGroups.get(k.getAdGroupId()).getType() != AdGroupType.CPM_BANNER
                                && simpleAdGroups.get(k.getAdGroupId()).getType() != AdGroupType.CPM_GEOPRODUCT);

                statDataByKeywordIds = getStatisticByKeywordIds(keywordsForRequest);
                pokazometerData = getPokazometerData(clientId, keywordsForRequest);

                // У нас есть приходящие с фронта ключевики для добавления. В них есть rowHash, но нет нормальной формы.
                // После добавления в операции добавляения просчитываются нормальные формы, но теряются rowHash данные.
                // Чтобы сохранить и то и другое для логивания в ОКР сохраняем все в отдельном хеше:
                // где ключ - rowHash строка, значение - сохраненный Keyword.
                List<AddedKeywordInfo> addResults = keywordsModificationResult.getResult().getAddResults();
                // Если вообще ничего не добавлено, тогда и не логировать.
                if (addResults != null) {
                    Map<Long, Keyword> keywords = listToMap(modifiedKeywords, Keyword::getId);
                    List<String> rowHashes = FunctionalUtils.mapList(pairsKeywordsToAdd, Pair::getKey);
                    Map<String, Keyword> rowHashToKeyword = new HashMap<>();
                    for (int index = 0; index < addResults.size(); index++) {
                        AddedKeywordInfo addedKeywordInfo = addResults.get(index);
                        if (addedKeywordInfo.isAdded()) {
                            Keyword savedKeyword = keywords.get(addedKeywordInfo.getId());
                            String rowHash = rowHashes.get(index);
                            rowHashToKeyword.put(rowHash, savedKeyword);
                        }
                    }
                    // Логирование в ОКР
                    adQualityExportLogger.logAddedKeywords(rowHashToKeyword);
                }
            }
        }

        Collection<Long> mainBids = request.getMainBids();
        if (mainBids.isEmpty()) {
            // Раз с фронтенда не пришли номера баннеров, значит будем вычислять сами
            Set<Long> adGroupIds = request.getAdGroupIds();
            mainBids = bannerRelationsRepository.getMainBannerIdsByAdGroupIds(clientId, adGroupIds).values();
        }

        List<BannerItemResponse> bannerItemsResponse =
                bannerStatusService.getBannersResponse(clientId, mainBids);

        // converting validation result from core to intapi

        ModifyIntapiValidationResult keywordsIntapiValidationResult = showConditionsValidationResultConverter
                .convertKeywordsResult(keywordsModificationResult, keywordsToAdd, keywordsToUpdate, keywordsToDelete);

        ModifyIntapiValidationResult relevanceMatchesIntapiValidationResult = showConditionsValidationResultConverter
                .convertRelevanceMatchesResult(relevanceMatchModificationResult,
                        relevanceMatchesToUpdate,
                        relevanceMatchesToDelete);

        ModifyIntapiValidationResult retargetingsIntapiValidationResult = showConditionsValidationResultConverter
                .convertRetargetingsResult(retargetingIdToAdGroupId,
                        retargetingsUpdateResult,
                        retargetingsDeleteResult);

        ModifyIntapiValidationResult searchRetargetingsIntapiValidationResult = showConditionsValidationResultConverter
                .convertRetargetingsResult(searchRetargetingIdToAdGroupId,
                        null,
                        searchRetargetingDeleteResult);

        ModifyIntapiValidationResult targetInterestsIntapiValidationResult = showConditionsValidationResultConverter
                .convertRetargetingsResult(targetInterestsIdToAdGroupId,
                        targetInterestsUpdateResult,
                        targetInterestsDeleteResult);

        return ShowConditionsResponse.executedOperation(isComplexOperationValid, clientCurrency)
                .addKeywordResult(keywordsModificationResult, keywordsIntapiValidationResult, keywordsModifyOperation,
                        inputKeywordPhrasesToAddMap, keywordsToUpdate,
                        modifiedKeywords, statDataByKeywordIds, pokazometerData)
                .addRelevanceMatchResult(relevanceMatchModificationResult, relevanceMatchesIntapiValidationResult)
                .addRetargetingResult(retargetingIdToAdGroupId, retargetingsUpdateResult, retargetingsDeleteResult,
                        retargetingsIntapiValidationResult)
                .addSearchRetargetingResult(searchRetargetingIdToAdGroupId, searchRetargetingDeleteResult,
                        searchRetargetingsIntapiValidationResult)
                .addTargetInterestsResult(targetInterestsIdToAdGroupId,
                        targetInterestsUpdateResult, targetInterestsDeleteResult, targetInterestsIntapiValidationResult)
                .addBannersStatusesResult(bannerItemsResponse);
    }

    private List<KeywordBidPokazometerData> getPokazometerData(ClientId clientId, List<Keyword> keywords) {
        return isEmpty(keywords)
                ? emptyList()
                : keywordBidDynamicDataService.safeGetPokazometerData(clientId, keywords);
    }

    private List<Keyword> getModifiedKeywords(ClientId clientId, Result<KeywordsModificationResult> result) {
        if (result == null || result.getResult() == null) {
            return null;
        }

        List<Long> keywordIds = new ArrayList<>();

        List<Long> addKeywordIds = mapList(result.getResult().getAddResults(), AddedKeywordInfo::getId);
        if (addKeywordIds != null) {
            keywordIds.addAll(addKeywordIds);
        }

        List<Long> updateKeywordIds = mapList(result.getResult().getUpdateResults(), UpdatedKeywordInfo::getId);
        if (updateKeywordIds != null) {
            keywordIds.addAll(updateKeywordIds);
        }

        return keywordService.getKeywords(clientId, keywordIds);
    }

    /**
     * Получение статистики по ключевым фразам, в случае какой-нибудь ошибки при чтении отдаётся пустая мапа.
     */
    private Map<Long, StatValueAggregator> getStatisticByKeywordIds(List<Keyword> keywords) {
        List<PhraseStatisticsRequest> phraseStatisticRequests = mapList(keywords,
                keyword -> new PhraseStatisticsRequest.Builder()
                        .withCampaignId(keyword.getCampaignId())
                        .withAdGroupId(keyword.getAdGroupId())
                        .withPhraseId(keyword.getId())
                        .build());
        DateRange dateRange = getDateRangeFromPeriod(Period.ofDays(DEFAULT_STATISTIC_PERIOD));

        try {
            Map<PhraseStatisticsResponse.ShowConditionStatIndex, StatValueAggregator> phraseStatistics =
                    recentStatisticsService.getPhraseStatistics(phraseStatisticRequests, dateRange,
                            PhraseStatisticsResponse::showConditionStatIndex);

            return EntryStream.of(phraseStatistics)
                    .mapKeys(PhraseStatisticsResponse.ShowConditionStatIndex::getPhraseId)
                    .toMap();

        } catch (RuntimeException e) {
            logger.info("Catch YT statisting getting error. Return empty map", e);
            return Collections.emptyMap();
        }
    }
}
