package ru.yandex.direct.intapi.entity.showconditions.service.validation;

import java.util.List;
import java.util.Map;
import java.util.function.BiFunction;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.keyword.container.KeywordsModificationResult;
import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.core.entity.relevancematch.container.RelevanceMatchModification;
import ru.yandex.direct.core.entity.relevancematch.container.RelevanceMatchModificationResult;
import ru.yandex.direct.core.entity.relevancematch.model.RelevanceMatch;
import ru.yandex.direct.core.entity.retargeting.model.Retargeting;
import ru.yandex.direct.intapi.validation.kernel.ValidationResultConversionService;
import ru.yandex.direct.intapi.validation.model.IntapiValidationResult;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.PathNode;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Preconditions.checkState;
import static java.util.stream.Collectors.toMap;
import static ru.yandex.direct.core.entity.keyword.container.KeywordsModificationContainer.ADD_LIST_NAME;
import static ru.yandex.direct.core.entity.keyword.container.KeywordsModificationContainer.COPY_AD_GROUPS_LIST_NAME;
import static ru.yandex.direct.core.entity.keyword.container.KeywordsModificationContainer.DELETE_LIST_NAME;
import static ru.yandex.direct.core.entity.keyword.container.KeywordsModificationContainer.UPDATE_LIST_NAME;
import static ru.yandex.direct.validation.result.PathHelper.field;

@Service
public class ShowConditionsValidationResultConverter {

    @Autowired
    private final ValidationResultConversionService validationResultConversionService;

    public ShowConditionsValidationResultConverter(
            ValidationResultConversionService validationResultConversionService) {
        this.validationResultConversionService = validationResultConversionService;
    }

    /**
     * Преобразование результата валидации операции в формат intapi
     *
     * @param result                            - результат операции модфикации КФ
     * @param keywordsToAdd                     - список КФ для добавления
     * @param adGroupIdToKeywordChangesToUpdate - список пар "идентификатор группы - изменения КФ" для обновления
     * @param adGroupIdToKeywordIdsToDelete     - список пар "идентификатор группы - идентификатор КФ" для удаления
     * @return контейнер с результатом валидации КФ в intapi формате
     */
    public ModifyIntapiValidationResult convertKeywordsResult(
            Result<KeywordsModificationResult> result,
            List<Keyword> keywordsToAdd,
            List<Pair<Long, ModelChanges<Keyword>>> adGroupIdToKeywordChangesToUpdate,
            List<Pair<Long, Long>> adGroupIdToKeywordIdsToDelete
    ) {
        ModifyIntapiValidationResult intapiValidationResult = new ModifyIntapiValidationResult();

        if (result == null) {
            return intapiValidationResult;
        }

        ValidationResult<?, Defect> vrAdd = result.getValidationResult().getSubResults().get(field(ADD_LIST_NAME));
        if (vrAdd != null) {
            Map<Integer, Keyword> keywordsToAddMap = EntryStream.of(keywordsToAdd).toMap();

            BiFunction<Integer, ValidationResult<?, Defect>, Long> adGroupGetter =
                    (index, itemVr) -> keywordsToAddMap.get(index).getAdGroupId();

            intapiValidationResult
                    .withAddValidationInfo(toValidationInfoMapByIndexes(vrAdd, adGroupGetter));
        }

        ValidationResult<?, Defect> vrUpdate =
                result.getValidationResult().getSubResults().get(field(UPDATE_LIST_NAME));
        if (vrUpdate != null) {
            Map<Integer, Pair<Long, ModelChanges<Keyword>>> keywordsToUpdateMap =
                    EntryStream.of(adGroupIdToKeywordChangesToUpdate).toMap();

            BiFunction<Integer, ValidationResult<?, Defect>, Long> adGroupGetter = (index, itemVr) -> {
                Pair<Long, ModelChanges<Keyword>> modelChangesPair = keywordsToUpdateMap.get(index);
                checkNotNull(modelChangesPair, "keyword must be in update map");
                return modelChangesPair.getLeft();
            };

            intapiValidationResult
                    .withUpdateValidationInfo(toValidationInfoMapByIndexes(vrUpdate, adGroupGetter));
        }

        ValidationResult<?, Defect> vrDelete =
                result.getValidationResult().getSubResults().get(field(DELETE_LIST_NAME));
        if (vrDelete != null) {

            Map<Long, Long> adGroupIdByKeywordIdToDeleteMap = StreamEx.of(adGroupIdToKeywordIdsToDelete)
                    .mapToEntry(Pair::getRight, Pair::getLeft)
                    .distinctKeys()
                    .toMap();

            BiFunction<Integer, ValidationResult<?, Defect>, Long> adGroupGetter = (index, itemVr) -> {
                Long keywordId = (Long) itemVr.getValue();
                Long adGroupId = adGroupIdByKeywordIdToDeleteMap.get(keywordId);
                checkNotNull(adGroupId, "keyword must be in delete map");
                return adGroupId;
            };

            intapiValidationResult
                    .withDeleteValidationInfo(toValidationInfoMapByIndexes(vrDelete, adGroupGetter));
        }

        ValidationResult<?, Defect> vrCopyAdGroups =
                result.getValidationResult().getSubResults().get(field(COPY_AD_GROUPS_LIST_NAME));
        if (vrCopyAdGroups != null) {
            IntapiValidationResult intapiVr =
                    validationResultConversionService.buildValidationResponse(vrCopyAdGroups).validationResult();
            intapiValidationResult.withCopyAdGroupsError(intapiVr.getErrors());
        }

        return intapiValidationResult;
    }

    /**
     * Преобразование результата валидации операции в формат intapi
     *
     * @param result                                   - результат операции модификации автотаргетингов
     * @param adGroupIdTorelevanceMatchChangesToUpdate - список пар "идентификатор группы - изменения
     *                                                 автотаргетингов" для обновления
     * @param adGroupIdToRelevanceMatchIdsToDelete     - cписок пар "идентификатор группы - идентификатор
     *                                                 автотаргетинга" для удаления
     * @return контейнер с результатом валидации автотаргетингов в intapi формате
     */
    public ModifyIntapiValidationResult convertRelevanceMatchesResult(
            Result<RelevanceMatchModificationResult> result,
            List<Pair<Long, ModelChanges<RelevanceMatch>>> adGroupIdTorelevanceMatchChangesToUpdate,
            List<Pair<Long, Long>> adGroupIdToRelevanceMatchIdsToDelete) {
        ModifyIntapiValidationResult intapiValidationResult = new ModifyIntapiValidationResult();

        if (result == null) {
            return intapiValidationResult;
        }

        ValidationResult<?, Defect> vrAdd = result.getValidationResult().getSubResults()
                .get(field(RelevanceMatchModification.RELEVANCE_MATCH_ADD.name()));
        if (vrAdd != null) {
            BiFunction<Integer, ValidationResult<?, Defect>, Long> adGroupGetter = (index, itemVr) -> {
                RelevanceMatch relevanceMatch = (RelevanceMatch) itemVr.getValue();
                Long adGroupId = relevanceMatch.getAdGroupId();
                checkNotNull(adGroupId, "relevance match adGroupId must be on add operation");
                return adGroupId;
            };

            intapiValidationResult.withAddValidationInfo(toValidationInfoMapByIndexes(vrAdd, adGroupGetter));
        }

        ValidationResult<?, Defect> vrUpdate = result.getValidationResult().getSubResults()
                .get(field(RelevanceMatchModification.RELEVANCE_MATCH_UPDATE.name()));
        if (vrUpdate != null) {
            Map<Integer, Pair<Long, ModelChanges<RelevanceMatch>>> relevanceMatchesToUpdateMap =
                    EntryStream.of(adGroupIdTorelevanceMatchChangesToUpdate).toMap();

            BiFunction<Integer, ValidationResult<?, Defect>, Long> adGroupGetter = (index, itemVr) -> {
                Pair<Long, ModelChanges<RelevanceMatch>> modelChangesPair = relevanceMatchesToUpdateMap.get(index);
                checkNotNull(modelChangesPair, "relevance match must be in update map");
                return modelChangesPair.getLeft();
            };

            intapiValidationResult.withUpdateValidationInfo(toValidationInfoMapByIndexes(vrUpdate, adGroupGetter));
        }

        ValidationResult<?, Defect> vrDelete = result.getValidationResult()
                .getSubResults().get(field(RelevanceMatchModification.RELEVANCE_MATCH_IDS_DELETE.name()));
        if (vrDelete != null) {
            Map<Long, Long> adGroupIdRelevanceMatchIdToDeleteMap = StreamEx.of(adGroupIdToRelevanceMatchIdsToDelete)
                    .mapToEntry(Pair::getRight, Pair::getLeft)
                    .distinctKeys()
                    .toMap();

            BiFunction<Integer, ValidationResult<?, Defect>, Long> adGroupGetter = (index, itemVr) -> {
                Long relevanceMatchId = (Long) itemVr.getValue();
                Long adGroupId = adGroupIdRelevanceMatchIdToDeleteMap.get(relevanceMatchId);
                checkNotNull(adGroupId, "relevance match must be in delete map");
                return adGroupId;
            };

            intapiValidationResult.withUpdateValidationInfo(toValidationInfoMapByIndexes(vrDelete, adGroupGetter));
        }

        return intapiValidationResult;
    }

    /**
     * Преобразование результата валидации операции в формат intapi
     *
     * @param retargetingIdToAdGroupId - мапа идентификаторов групп по идентификаторам ретаргетингов
     *                                 (таргетов по интересам)
     * @param resultUpdate             - результат операции обновления
     * @param resultDelete             - результат операции удаления
     * @return контейнер с результатом валидации ретаргетингов (таргетов по интересам) в intapi формате
     */
    public ModifyIntapiValidationResult convertRetargetingsResult(
            Map<Long, Long> retargetingIdToAdGroupId,
            MassResult<Long> resultUpdate,
            MassResult<Long> resultDelete) {
        ModifyIntapiValidationResult intapiValidationResult = new ModifyIntapiValidationResult();

        ValidationResult<?, Defect> vrUpdate = resultUpdate != null ? resultUpdate.getValidationResult() : null;
        if (vrUpdate != null) {
            BiFunction<Integer, ValidationResult<?, Defect>, Long> adGroupGetter = (index, itemVr) -> {
                checkState(itemVr.getValue() instanceof Retargeting,
                        "retargeting update validation should be performed on retargeting model");
                return retargetingIdToAdGroupId.get(((Retargeting) itemVr.getValue()).getId());
            };

            intapiValidationResult.withUpdateValidationInfo(toValidationInfoMapByIndexes(vrUpdate, adGroupGetter));
        }

        ValidationResult<?, Defect> vrDelete = resultDelete != null ? resultDelete.getValidationResult() : null;
        if (vrDelete != null) {
            BiFunction<Integer, ValidationResult<?, Defect>, Long> adGroupGetter = (index, itemVr) -> {
                checkState(itemVr.getValue() instanceof Long,
                        "retargetings delete validation should be performed on long identifier");
                Long id = (Long) itemVr.getValue();
                return retargetingIdToAdGroupId.get(id);
            };

            intapiValidationResult.withDeleteValidationInfo(toValidationInfoMapByIndexes(vrDelete, adGroupGetter));
        }

        return intapiValidationResult;
    }

    private Map<Integer, ModifyIntapiValidationResult.ValidationContainer> toValidationInfoMapByIndexes(
            ValidationResult<?, Defect> validationResult,
            BiFunction<Integer, ValidationResult<?, Defect>, Long> adGroupIdGetter) {
        if (validationResult == null) {
            return null;
        }

        return EntryStream.of(validationResult.getSubResults())
                .mapKeys(pn -> ((PathNode.Index) pn).getIndex())
                .collect(toMap(Map.Entry::getKey, e -> {
                    Integer index = e.getKey();
                    ValidationResult<?, Defect> vr = e.getValue();
                    Long adGroupId = adGroupIdGetter.apply(index, vr);

                    IntapiValidationResult intapiVr =
                            validationResultConversionService.buildValidationResponse(vr).validationResult();

                    return new ModifyIntapiValidationResult.ValidationContainer(adGroupId, intapiVr.getErrors());
                }));
    }
}


