package ru.yandex.direct.intapi.entity.showconditions.service.validation;

import java.util.Set;

import org.apache.commons.lang3.tuple.Pair;
import org.springframework.stereotype.Service;

import ru.yandex.direct.intapi.entity.showconditions.model.request.ShowConditionsRequest;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.Collections.disjoint;
import static ru.yandex.direct.intapi.entity.showconditions.service.validation.ShowConditionsDefects.duplicatedKeywordUsage;
import static ru.yandex.direct.intapi.entity.showconditions.service.validation.ShowConditionsDefects.duplicatedRelevanceMatchUsage;
import static ru.yandex.direct.intapi.entity.showconditions.service.validation.ShowConditionsDefects.duplicatedRetargetingUsage;
import static ru.yandex.direct.intapi.entity.showconditions.service.validation.ShowConditionsDefects.duplicatedTargetInterestsUsage;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;

@Service
public class ShowConditionsValidationService {

    public ValidationResult<ShowConditionsRequest, Defect> validate(ShowConditionsRequest request) {
        return ItemValidationBuilder.of(request, Defect.class)
                .check(updatedAndDeletedKeywordsAreDifferent())
                .check(updatedAndDeletedRelevanceMatchesAreDifferent())
                .check(updatedAndDeletedRetargetingsAreDifferent())
                .check(updatedAndDeletedTargetInterestsAreDifferent())
                .getResult();
    }

    private Constraint<ShowConditionsRequest, Defect> updatedAndDeletedKeywordsAreDifferent() {
        return fromPredicate(r -> {
                    Set<Long> idsToUpdate = listToSet(r.getKeywordChangesToEdit(), pair -> pair.getRight().getId());
                    return disjoint(idsToUpdate, mapList(r.getAdGroupIdToKeywordToDelete(), Pair::getRight));
                },
                duplicatedKeywordUsage()
        );
    }

    private Constraint<ShowConditionsRequest, Defect> updatedAndDeletedRetargetingsAreDifferent() {
        return fromPredicate(r -> {
                    Set<Long> idsToUpdate = listToSet(r.getRetargetingChangesToEdit(), ModelChanges::getId);
                    return disjoint(idsToUpdate, r.getRetargetingIdsToDelete());
                },
                duplicatedRetargetingUsage()
        );
    }

    private Constraint<ShowConditionsRequest, Defect> updatedAndDeletedTargetInterestsAreDifferent() {
        return fromPredicate(r -> {
                    Set<Long> idsToUpdate = listToSet(r.getTargetInterestsChangesToEdit(), ModelChanges::getId);
                    return disjoint(idsToUpdate, r.getTargetInterestsIdsToDelete());
                },
                duplicatedTargetInterestsUsage()
        );
    }

    private Constraint<ShowConditionsRequest, Defect> updatedAndDeletedRelevanceMatchesAreDifferent() {
        return fromPredicate(r -> {
                    Set<Long> idsToUpdate = listToSet(r.getRelevanceMatchChangesToEdit(), p -> p.getRight().getId());
                    return disjoint(idsToUpdate, mapList(r.getAdGroupIdToRelevanceMatchToDelete(), Pair::getRight));
                },
                duplicatedRelevanceMatchUsage()
        );
    }
}
