package ru.yandex.direct.intapi.entity.statistic;

import java.math.BigDecimal;
import java.util.List;
import java.util.Map;

import javax.ws.rs.GET;
import javax.ws.rs.POST;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import one.util.streamex.EntryStream;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.direct.core.entity.statistics.model.FraudAndGiftClicks;
import ru.yandex.direct.core.entity.statistics.model.StatisticsDateRange;
import ru.yandex.direct.core.entity.statistics.service.OrderStatService;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.currency.Money;
import ru.yandex.direct.intapi.ErrorResponse;
import ru.yandex.direct.intapi.entity.statistic.model.GetOrderClicksTodayResponse;
import ru.yandex.direct.intapi.entity.statistic.model.GetSessionsNumForWeekResponse;
import ru.yandex.direct.intapi.entity.statistic.model.dynamic.GetDynamicStatisticResponseItem;
import ru.yandex.direct.intapi.entity.statistic.model.dynamic.GetDynamicStatisticsRequest;
import ru.yandex.direct.intapi.entity.statistic.model.order.GetOrdersDaysNumResponse;
import ru.yandex.direct.intapi.entity.statistic.model.order.GetOrdersStatByIntervalRequest;
import ru.yandex.direct.intapi.entity.statistic.model.order.GetOrdersStatRequest;
import ru.yandex.direct.intapi.entity.statistic.model.order.GetOrdersSumSpentRequest;
import ru.yandex.direct.intapi.entity.statistic.model.performance.GetPerformanceStatisticResponseItem;
import ru.yandex.direct.intapi.entity.statistic.model.performance.GetPerformanceStatisticsRequest;
import ru.yandex.direct.intapi.entity.statistic.model.phrase.GetPhraseStatisticResponseItem;
import ru.yandex.direct.intapi.entity.statistic.model.phrase.GetPhraseStatisticsRequest;
import ru.yandex.direct.intapi.entity.statistic.model.relevancematch.GetRelevanceMatchStatisticResponseItem;
import ru.yandex.direct.intapi.entity.statistic.model.relevancematch.GetRelevanceMatchStatisticsRequest;
import ru.yandex.direct.intapi.entity.statistic.model.retargeting.GetRetargetingStatisticResponseItem;
import ru.yandex.direct.intapi.entity.statistic.model.retargeting.GetRetargetingStatisticsRequest;
import ru.yandex.direct.intapi.entity.statistic.service.StatisticService;
import ru.yandex.direct.intapi.entity.statistic.service.StatisticValidationService;
import ru.yandex.direct.intapi.validation.IntApiDefect;
import ru.yandex.direct.tvm.AllowServices;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;

import static ru.yandex.direct.intapi.validation.IntApiConstraints.validId;
import static ru.yandex.direct.intapi.validation.ValidationUtils.checkResult;
import static ru.yandex.direct.tvm.TvmService.DIRECT_API_PROD;
import static ru.yandex.direct.tvm.TvmService.DIRECT_API_TEST;
import static ru.yandex.direct.tvm.TvmService.DIRECT_DEVELOPER;
import static ru.yandex.direct.tvm.TvmService.DIRECT_INTAPI_PROD;
import static ru.yandex.direct.tvm.TvmService.DIRECT_INTAPI_TEST;
import static ru.yandex.direct.tvm.TvmService.DIRECT_SCRIPTS_PROD;
import static ru.yandex.direct.tvm.TvmService.DIRECT_SCRIPTS_TEST;
import static ru.yandex.direct.tvm.TvmService.DIRECT_WEB_PROD;
import static ru.yandex.direct.tvm.TvmService.DIRECT_WEB_TEST;

/**
 * Ручки для получения статистики
 */
@RestController
@RequestMapping("statistic")
@Api(tags = "statistic")
@AllowServices(production = {DIRECT_WEB_PROD, DIRECT_API_PROD, DIRECT_SCRIPTS_PROD, DIRECT_INTAPI_PROD},
        testing = {DIRECT_WEB_TEST, DIRECT_API_TEST, DIRECT_SCRIPTS_TEST, DIRECT_INTAPI_TEST, DIRECT_DEVELOPER},
        sandboxTesting = {DIRECT_DEVELOPER})
public class StatisticController {
    private final StatisticService statisticService;
    private final OrderStatService orderStatService;
    private final StatisticValidationService statisticValidationService;

    @Autowired
    public StatisticController(StatisticService statisticService,
                               OrderStatService orderStatService,
                               StatisticValidationService statisticValidationService) {
        this.statisticService = statisticService;
        this.orderStatService = orderStatService;
        this.statisticValidationService = statisticValidationService;
    }

    @POST
    @ApiOperation(
            value = "getPhraseStatistic",
            httpMethod = "POST",
            nickname = "getPhraseStatistic"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 400, message = "Bad params", response = ErrorResponse.class),
                    @ApiResponse(code = 200, message = "Ok", response = List.class)
            }
    )
    @RequestMapping(path = "/phrase/get",
            method = RequestMethod.POST,
            consumes = MediaType.APPLICATION_JSON_UTF8_VALUE,
            produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    @ResponseBody
    public List<GetPhraseStatisticResponseItem> getPhraseStatistic(@RequestBody GetPhraseStatisticsRequest request) {
        return statisticService.getPhraseStatistic(request);
    }

    @POST
    @ApiOperation(
            value = "getRelevanceMatchStatistic",
            httpMethod = "POST",
            nickname = "getRelevanceMatchStatistic"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 400, message = "Bad params", response = ErrorResponse.class),
                    @ApiResponse(code = 200, message = "Ok", response = List.class)
            }
    )
    @RequestMapping(path = "/relevance_match/get",
            method = RequestMethod.POST,
            consumes = MediaType.APPLICATION_JSON_UTF8_VALUE,
            produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    @ResponseBody
    public List<GetRelevanceMatchStatisticResponseItem> getRelevanceMatchStatistic(
            @RequestBody GetRelevanceMatchStatisticsRequest request) {
        return statisticService.getRelevanceMatchStatistic(request);
    }

    @POST
    @ApiOperation(
            value = "getRetargetingStatistic",
            httpMethod = "POST",
            nickname = "getRetargetingStatistic"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 400, message = "Bad params", response = ErrorResponse.class),
                    @ApiResponse(code = 200, message = "Ok", response = List.class)
            }
    )
    @RequestMapping(path = "/retargeting/get",
            method = RequestMethod.POST,
            consumes = MediaType.APPLICATION_JSON_UTF8_VALUE,
            produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    @ResponseBody
    public List<GetRetargetingStatisticResponseItem> getRetargetingStatistic(
            @RequestBody GetRetargetingStatisticsRequest request) {
        return statisticService.getRetargetingStatistic(request);
    }

    @POST
    @ApiOperation(
            value = "getDynamicStatistic",
            httpMethod = "POST",
            nickname = "getDynamicStatistic"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 400, message = "Bad params", response = ErrorResponse.class),
                    @ApiResponse(code = 200, message = "Ok", response = List.class)
            }
    )
    @RequestMapping(path = "/dynamic/get",
            method = RequestMethod.POST,
            consumes = MediaType.APPLICATION_JSON_UTF8_VALUE,
            produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    @ResponseBody
    public List<GetDynamicStatisticResponseItem> getDynamicStatistic(
            @RequestBody GetDynamicStatisticsRequest request) {
        return statisticService.getDynamicStatistic(request);
    }

    @GET
    @ApiOperation(
            value = "getSessionsNumForWeek",
            httpMethod = "GET",
            nickname = "getSessionsNumForWeek"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 400, message = "Bad params", response = ErrorResponse.class),
                    @ApiResponse(code = 200, message = "Ok", response = GetSessionsNumForWeekResponse.class)
            }
    )
    @RequestMapping(path = "/order/getSessionsNumForWeek",
            method = RequestMethod.GET)
    @ResponseBody
    public GetSessionsNumForWeekResponse getSessionsNumForWeek(@RequestParam(value = "order_id") Long orderId) {

        ItemValidationBuilder<Long, IntApiDefect> v =
                ItemValidationBuilder.of(orderId, IntApiDefect.class);
        checkResult(v.check(validId()).getResult());
        Long sessionNum = orderStatService.getOrderSessionsNumForWeek(orderId);
        return new GetSessionsNumForWeekResponse(sessionNum);
    }

    @GET
    @ApiOperation(
            value = "getOrderClicksToday",
            httpMethod = "GET",
            nickname = "getOrderClicksToday"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 400, message = "Bad params", response = ErrorResponse.class),
                    @ApiResponse(code = 200, message = "Ok", response = GetOrderClicksTodayResponse.class)
            }
    )
    @GetMapping(path = "/order/getOrderClicksToday")
    @ResponseBody
    public GetOrderClicksTodayResponse getOrderClicksToday(@RequestParam(value = "order_id") Long orderId) {

        ItemValidationBuilder<Long, IntApiDefect> v =
                ItemValidationBuilder.of(orderId, IntApiDefect.class);
        checkResult(v.check(validId()).getResult());
        Long clicks = orderStatService.getOrderClicksToday(orderId);
        return new GetOrderClicksTodayResponse(clicks);
    }

    @POST
    @ApiOperation(
            value = "getOrdersFirstLastDay",
            httpMethod = "POST",
            nickname = "getOrdersFirstLastDay"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 400, message = "Bad params", response = ErrorResponse.class),
                    @ApiResponse(code = 200, message = "Ok", response = List.class)
            }
    )
    @PostMapping(path = "/orders/getOrdersFirstLastDay",
            consumes = MediaType.APPLICATION_JSON_UTF8_VALUE,
            produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    @ResponseBody
    public Map<Long, StatisticsDateRange> getOrdersStartLastDay(@RequestBody GetOrdersStatRequest request) {
        ItemValidationBuilder<GetOrdersStatRequest, IntApiDefect> vb = ItemValidationBuilder.of(request);
        vb.list(request.getOrderIds(), GetOrdersStatRequest.ORDER_IDS).checkEach(validId());
        checkResult(vb.getResult());
        return orderStatService.getOrdersFirstLastDay(request.getOrderIds());
    }

    @POST
    @ApiOperation(
            value = "Количество дней за которые есть статистика",
            httpMethod = "POST",
            nickname = "getOrdersDaysNum"
    )
    @PostMapping(path = "/orders/getOrdersDaysNum",
            consumes = MediaType.APPLICATION_JSON_UTF8_VALUE,
            produces = MediaType.APPLICATION_JSON_UTF8_VALUE
    )
    @ResponseBody
    public GetOrdersDaysNumResponse getOrdersDaysNum(@RequestBody GetOrdersStatByIntervalRequest request) {
        statisticValidationService.validateAndCheck(request);
        Long ordersDaysNum = orderStatService.getOrdersDaysNum(
                request.getOrderIds(), request.getStartDate(), request.getEndDate());
        return new GetOrdersDaysNumResponse(ordersDaysNum);
    }

    @POST
    @ApiOperation(
            value = "getPerformanceStatistic",
            httpMethod = "POST",
            nickname = "getPerformanceStatistic"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 400, message = "Bad params", response = ErrorResponse.class),
                    @ApiResponse(code = 200, message = "Ok", response = List.class)
            }
    )
    @RequestMapping(path = "/performance/get",
            method = RequestMethod.POST,
            consumes = MediaType.APPLICATION_JSON_UTF8_VALUE,
            produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    @ResponseBody
    public List<GetPerformanceStatisticResponseItem> getPerformanceStatistic(
            @RequestBody GetPerformanceStatisticsRequest request) {
        return statisticService.getPerformanceStatistic(request);
    }

    @POST
    @ApiOperation(
            value = "Суммарные данные по подаренным и недействительным кликам для набора заказов",
            httpMethod = "POST",
            nickname = "getFraudClicks"
    )
    @PostMapping(path = "/orders/getFraudClicks",
            consumes = MediaType.APPLICATION_JSON_UTF8_VALUE,
            produces = MediaType.APPLICATION_JSON_UTF8_VALUE
    )
    @ResponseBody
    public FraudAndGiftClicks getFraudClicks(@RequestBody GetOrdersStatByIntervalRequest request) {
        statisticValidationService.validateAndCheck(request);
        return orderStatService.getFraudClicks(request.getOrderIds(), request.getStartDate(), request.getEndDate());
    }

    @POST
    @ApiOperation(
            value = "Предсказание трат по кампании",
            httpMethod = "POST",
            nickname = "getRoughForecast"
    )
    @PostMapping(path = "/campaigns/getRoughForecast",
            consumes = MediaType.APPLICATION_JSON_UTF8_VALUE,
            produces = MediaType.APPLICATION_JSON_UTF8_VALUE
    )
    @ResponseBody
    public Map<Long, BigDecimal> getRoughForecast(@RequestBody List<Long> campaignIds,
                                                  @RequestParam(value = "currency") CurrencyCode currencyCode) {
        statisticValidationService.validateAndCheckIds(campaignIds, "ids");
        return EntryStream.of(orderStatService.getCampBsstatForecast(campaignIds, currencyCode))
                .mapValues(Money::bigDecimalValue)
                .toMap();
    }

    @POST
    @ApiOperation(
            value = "Траты по заказам за периоды",
            httpMethod = "POST",
            nickname = "getSumSpent"
    )
    @PostMapping(path = "/orders/getSumSpent",
            consumes = MediaType.APPLICATION_JSON_UTF8_VALUE,
            produces = MediaType.APPLICATION_JSON_UTF8_VALUE
    )
    @ResponseBody
    public Map<String, BigDecimal> getOrdersSumSpent(@RequestBody GetOrdersSumSpentRequest request,
                                                     @RequestParam(value = "currency") CurrencyCode currencyCode) {
        statisticValidationService.validateAndCheckIds(request.getOrderIds(), GetOrdersSumSpentRequest.ORDER_IDS_FIELD);
        statisticValidationService.validateAndCheckPeriods(
                request.getPeriods(), GetOrdersSumSpentRequest.PERIODS_FIELD);
        return EntryStream.of(orderStatService.getOrdersSumSpent(
                request.getOrderIds(), request.getPeriods(), currencyCode))
                .mapValues(Money::bigDecimalValue)
                .toMap();
    }


    @POST
    @ApiOperation(
            value = "Потрачено сегодня",
            httpMethod = "POST",
            nickname = "getOrdersSpentToday"
    )
    @PostMapping(path = "/orders/getSpentToday",
            consumes = MediaType.APPLICATION_JSON_UTF8_VALUE,
            produces = MediaType.APPLICATION_JSON_UTF8_VALUE
    )
    @ResponseBody
    public Map<Long, BigDecimal> getOrdersSpentToday(@RequestBody List<Long> orderIds,
                                                     @RequestParam(value = "with_nds") Boolean withNds) {
        statisticValidationService.validateAndCheckIds(orderIds, "ids");
        return EntryStream.of(orderStatService.getOrdersSpentToday(orderIds, withNds))
                .mapValues(Money::bigDecimalValue)
                .toMap();
    }
}
