package ru.yandex.direct.intapi.entity.statistic.service;

import java.time.Period;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import one.util.streamex.EntryStream;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.intapi.entity.statistic.model.PhraseStatisticConverter;
import ru.yandex.direct.intapi.entity.statistic.model.dynamic.GetDynamicStatisticResponseItem;
import ru.yandex.direct.intapi.entity.statistic.model.dynamic.GetDynamicStatisticsRequest;
import ru.yandex.direct.intapi.entity.statistic.model.performance.GetPerformanceStatisticResponseItem;
import ru.yandex.direct.intapi.entity.statistic.model.performance.GetPerformanceStatisticsRequest;
import ru.yandex.direct.intapi.entity.statistic.model.phrase.GetPhraseStatisticResponseItem;
import ru.yandex.direct.intapi.entity.statistic.model.phrase.GetPhraseStatisticsRequest;
import ru.yandex.direct.intapi.entity.statistic.model.relevancematch.GetRelevanceMatchStatisticResponseItem;
import ru.yandex.direct.intapi.entity.statistic.model.relevancematch.GetRelevanceMatchStatisticsRequest;
import ru.yandex.direct.intapi.entity.statistic.model.retargeting.GetRetargetingStatisticResponseItem;
import ru.yandex.direct.intapi.entity.statistic.model.retargeting.GetRetargetingStatisticsRequest;
import ru.yandex.direct.intapi.validation.IntApiDefect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.ytcomponents.statistics.model.DateRange;
import ru.yandex.direct.ytcomponents.statistics.model.PhraseStatisticsRequest;
import ru.yandex.direct.ytcomponents.statistics.model.PhraseStatisticsResponse;
import ru.yandex.direct.ytcomponents.statistics.model.RetargetingStatisticsRequest;
import ru.yandex.direct.ytcomponents.statistics.model.ShowConditionStatisticsRequest;
import ru.yandex.direct.ytcomponents.statistics.model.StatValueAggregator;
import ru.yandex.direct.ytcore.entity.statistics.service.RecentStatisticsService;

import static ru.yandex.direct.intapi.entity.statistic.model.PhraseStatisticConverter.getDateRangeFromPeriod;
import static ru.yandex.direct.intapi.entity.statistic.model.PhraseStatisticConverter.toYtRetargetingStatisticRequest;
import static ru.yandex.direct.intapi.entity.statistic.model.PhraseStatisticConverter.toYtShowConditionStatisticRequest;
import static ru.yandex.direct.intapi.validation.ValidationUtils.checkResult;

/**
 * Сервис для получения статистики о показах по различным условиям.
 * <p>
 * Данные берутся из YT {@code DirectPhraseStat}.
 * <p>
 * См. также: <a href="https://st.yandex-team.ru/DIRECT-78175">
 * (DIRECT-78175) Статистика за 28 календарных дней в Сетях и для разных типов условий показа (замена bsrank/ppstate)
 * </a>
 */
@Service
public class StatisticService {

    private final RecentStatisticsService recentStatisticsService;
    private final StatisticValidationService statisticValidationService;

    @Autowired
    public StatisticService(RecentStatisticsService recentStatisticsService,
                            StatisticValidationService statisticValidationService) {
        this.recentStatisticsService = recentStatisticsService;
        this.statisticValidationService = statisticValidationService;
    }

    /**
     * Возвращает статистику для фраз.
     * <p>
     * При её подсчёте в поисковой статистике учитывается только площадка "Яндекс"
     */
    public List<GetPhraseStatisticResponseItem> getPhraseStatistic(
            GetPhraseStatisticsRequest getPhraseStatisticRequest) {

        ValidationResult<GetPhraseStatisticsRequest, IntApiDefect> validationResult =
                statisticValidationService.validate(getPhraseStatisticRequest);
        checkResult(validationResult);

        Collection<PhraseStatisticsRequest> phraseStatisticRequests =
                PhraseStatisticConverter.toYtPhraseStatisticRequest(getPhraseStatisticRequest);
        DateRange dateRange = getDateRangeFromPeriod(Period.ofDays(getPhraseStatisticRequest.getIntervalDays()));

        Map<PhraseStatisticsResponse.PhraseStatIndex, StatValueAggregator> response =
                recentStatisticsService.getPhraseStatistics(phraseStatisticRequests, dateRange);

        return EntryStream.of(response)
                .mapKeyValue((key, stat) -> new GetPhraseStatisticResponseItem()
                        .withCampaignId(key.getCampaignId())
                        .withAdGroupId(key.getAdGroupId())
                        .withBannerId(key.getBannerId())
                        .withBsPhraseId(key.getBsPhraseId().toBigInteger())
                        .withPhraseId(key.getPhraseId())
                        .withClicks(stat.getSearchClicks())
                        .withShows(stat.getSearchShows())
                        .withEshows(stat.getSearchEshows())
                        .withContextClicks(stat.getNetworkClicks())
                        .withContextShows(stat.getNetworkShows())
                        .withContextEshows(stat.getNetworkEshows()))
                .toList();
    }

    /**
     * Возвращает статистику для автотаргетинга (relevance match)
     */
    public List<GetRelevanceMatchStatisticResponseItem> getRelevanceMatchStatistic(
            GetRelevanceMatchStatisticsRequest getStatisticRequest) {
        ValidationResult<GetRelevanceMatchStatisticsRequest, IntApiDefect> validationResult =
                statisticValidationService.validate(getStatisticRequest);
        checkResult(validationResult);

        Collection<ShowConditionStatisticsRequest> showConditionStatisticRequests =
                PhraseStatisticConverter.toYtShowConditionStatisticRequest(getStatisticRequest);
        DateRange dateRange = getDateRangeFromPeriod(Period.ofDays(getStatisticRequest.getIntervalDays()));

        Map<PhraseStatisticsResponse.ShowConditionStatIndex, StatValueAggregator> response =
                recentStatisticsService.getShowConditionStatistics(showConditionStatisticRequests, dateRange);

        return EntryStream.of(response)
                .mapKeyValue((key, stat) -> new GetRelevanceMatchStatisticResponseItem()
                        .withCampaignId(key.getCampaignId())
                        .withAdGroupId(key.getAdGroupId())
                        .withRelevanceMatchId(key.getPhraseId())
                        .withClicks(stat.getSearchClicks())
                        .withShows(stat.getSearchShows())
                        .withEshows(stat.getSearchEshows())
                        .withContextClicks(stat.getNetworkClicks())
                        .withContextShows(stat.getNetworkShows()))
                .toList();
    }

    /**
     * Возвращает статистику для retargeting'ов
     */
    public List<GetRetargetingStatisticResponseItem> getRetargetingStatistic(GetRetargetingStatisticsRequest request) {

        ValidationResult<GetRetargetingStatisticsRequest, IntApiDefect> validationResult =
                statisticValidationService.validate(request);
        checkResult(validationResult);

        Collection<RetargetingStatisticsRequest> retargetingStatisticRequests =
                toYtRetargetingStatisticRequest(request);
        DateRange dateRange = getDateRangeFromPeriod(Period.ofDays(request.getIntervalDays()));

        Map<PhraseStatisticsResponse.GoalContextStatIndex, StatValueAggregator> response =
                recentStatisticsService.getRetargetingStatistics(retargetingStatisticRequests, dateRange);

        return EntryStream.of(response)
                .mapKeyValue((key, stat) -> new GetRetargetingStatisticResponseItem()
                        .withCampaignId(key.getCampaignId())
                        .withAdGroupId(key.getAdGroupId())
                        .withRetargetingConditionId(key.getGoalContextId().longValue())
                        .withContextClicks(stat.getNetworkClicks())
                        .withContextShows(stat.getNetworkShows()))
                .toList();
    }

    /**
     * Возвращает статистику для динамических объявлений
     */
    public List<GetDynamicStatisticResponseItem> getDynamicStatistic(GetDynamicStatisticsRequest request) {
        ValidationResult<GetDynamicStatisticsRequest, IntApiDefect> validationResult =
                statisticValidationService.validate(request);
        checkResult(validationResult);

        Collection<ShowConditionStatisticsRequest> showConditionStatisticRequests =
                PhraseStatisticConverter.toYtShowConditionStatisticRequest(request);
        DateRange dateRange = getDateRangeFromPeriod(Period.ofDays(request.getIntervalDays()));

        Map<PhraseStatisticsResponse.ShowConditionStatIndex, StatValueAggregator> response =
                recentStatisticsService.getShowConditionStatistics(showConditionStatisticRequests, dateRange);

        return EntryStream.of(response)
                .mapKeyValue((key, stat) -> new GetDynamicStatisticResponseItem()
                        .withCampaignId(key.getCampaignId())
                        .withAdGroupId(key.getAdGroupId())
                        .withDynamicConditionId(key.getPhraseId())
                        .withClicks(stat.getSearchClicks())
                        .withShows(stat.getSearchShows())
                        .withEshows(stat.getSearchEshows())
                        .withContextClicks(stat.getNetworkClicks())
                        .withContextShows(stat.getNetworkShows()))
                .toList();
    }

    /**
     * Возвращает статистику для performance объявлений
     */
    public List<GetPerformanceStatisticResponseItem> getPerformanceStatistic(GetPerformanceStatisticsRequest request) {
        ValidationResult<GetPerformanceStatisticsRequest, IntApiDefect> validationResult =
                statisticValidationService.validate(request);
        checkResult(validationResult);

        Collection<ShowConditionStatisticsRequest> showConditionStatisticRequests =
                toYtShowConditionStatisticRequest(request);
        DateRange dateRange = getDateRangeFromPeriod(Period.ofDays(request.getIntervalDays()));

        Map<PhraseStatisticsResponse.ShowConditionStatIndex, StatValueAggregator> response =
                recentStatisticsService.getShowConditionStatistics(showConditionStatisticRequests, dateRange);

        return EntryStream.of(response)
                .mapKeyValue((statIndex, stat) -> new GetPerformanceStatisticResponseItem()
                        .withCampaignId(statIndex.getCampaignId())
                        .withAdGroupId(statIndex.getAdGroupId())
                        .withPerformanceFilterId(statIndex.getPhraseId())
                        .withClicks(stat.getSearchClicks())
                        .withShows(stat.getSearchShows())
                        .withEshows(stat.getSearchEshows())
                        .withContextClicks(stat.getNetworkClicks())
                        .withContextShows(stat.getNetworkShows()))
                .toList();
    }
}
