package ru.yandex.direct.intapi.entity.statistic.service;

import java.time.LocalDate;
import java.util.Collection;
import java.util.List;
import java.util.function.Predicate;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.statistics.model.Period;
import ru.yandex.direct.intapi.IntApiException;
import ru.yandex.direct.intapi.entity.statistic.model.dynamic.GetDynamicStatisticsRequest;
import ru.yandex.direct.intapi.entity.statistic.model.dynamic.GetDynamicStatisticsRequestItem;
import ru.yandex.direct.intapi.entity.statistic.model.order.GetOrdersStatByIntervalRequest;
import ru.yandex.direct.intapi.entity.statistic.model.performance.GetPerformanceStatisticsRequest;
import ru.yandex.direct.intapi.entity.statistic.model.performance.GetPerformanceStatisticsRequestItem;
import ru.yandex.direct.intapi.entity.statistic.model.phrase.GetPhraseStatisticsRequest;
import ru.yandex.direct.intapi.entity.statistic.model.relevancematch.GetRelevanceMatchStatisticsRequest;
import ru.yandex.direct.intapi.entity.statistic.model.relevancematch.GetRelevanceMatchStatisticsRequestItem;
import ru.yandex.direct.intapi.entity.statistic.model.retargeting.GetRetargetingStatisticsRequest;
import ru.yandex.direct.intapi.entity.statistic.model.retargeting.GetRetargetingStatisticsRequestItem;
import ru.yandex.direct.intapi.validation.IntApiDefect;
import ru.yandex.direct.intapi.validation.kernel.ValidationResultConversionService;
import ru.yandex.direct.intapi.validation.model.IntapiValidationResponse;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.constraint.CommonConstraints;
import ru.yandex.direct.validation.defect.CommonDefects;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.entity.statistics.service.OrderStatService.BEGIN_OF_TIME_FOR_STAT;
import static ru.yandex.direct.intapi.validation.IntApiConstraints.greaterThan;
import static ru.yandex.direct.intapi.validation.IntApiConstraints.notEmptyCollection;
import static ru.yandex.direct.intapi.validation.IntApiConstraints.notNull;
import static ru.yandex.direct.intapi.validation.IntApiConstraints.validId;
import static ru.yandex.direct.intapi.validation.IntApiDefects.notEmptyRequest;
import static ru.yandex.direct.validation.constraint.DateConstraints.isNotBeforeThan;

@Service
public class StatisticValidationService {

    private final ValidationResultConversionService validationResultConversionService;

    @Autowired
    public StatisticValidationService(ValidationResultConversionService validationResultConversionService) {
        this.validationResultConversionService = validationResultConversionService;
    }

    public ValidationResult<GetPhraseStatisticsRequest, IntApiDefect> validate(
            GetPhraseStatisticsRequest getPhraseStatisticsRequest) {
        ItemValidationBuilder<GetPhraseStatisticsRequest, IntApiDefect> v =
                ItemValidationBuilder.of(getPhraseStatisticsRequest, IntApiDefect.class);

        v.check(notNull(), notEmptyRequest());
        if (!v.getResult().hasAnyErrors()) {
            v.item(getPhraseStatisticsRequest.getIntervalDays(), GetPhraseStatisticsRequest.INTERVAL_DAYS)
                    .check(notNull())
                    .check(greaterThan(0));

            v.list(getPhraseStatisticsRequest.getGetPhraseStatisticsRequestItems(),
                    GetPhraseStatisticsRequest.SELECTION_CRITERIA)
                    .check(notNull())
                    .check(notEmptyCollection())
                    .checkEach(notNull());
        }
        return v.getResult();

    }

    /**
     * Проверка на корректность запроса на статистику для автотаргетинга
     */
    public ValidationResult<GetRelevanceMatchStatisticsRequest, IntApiDefect> validate(
            GetRelevanceMatchStatisticsRequest request) {
        ItemValidationBuilder<GetRelevanceMatchStatisticsRequest, IntApiDefect> v =
                ItemValidationBuilder.of(request, IntApiDefect.class);

        v.check(notNull(), notEmptyRequest());
        if (!v.getResult().hasAnyErrors()) {
            v.item(request.getIntervalDays(), GetRelevanceMatchStatisticsRequest.INTERVAL_DAYS)
                    .check(notNull())
                    .check(greaterThan(0));

            v.list(request.getGetRelevanceMatchStatisticsRequestItems(),
                    GetRelevanceMatchStatisticsRequest.SELECTION_CRITERIA)
                    .check(notNull())
                    .check(notEmptyCollection())
                    .checkEach(notNull())
                    .checkEachBy(relevanceMatchRequestItemValidator(), When.notNull());
        }
        return v.getResult();
    }

    /**
     * В {@link GetRelevanceMatchStatisticsRequestItem} все параметры должны быть заполнены валидными ID
     */
    private Validator<GetRelevanceMatchStatisticsRequestItem, IntApiDefect> relevanceMatchRequestItemValidator() {
        return item -> {
            ItemValidationBuilder<GetRelevanceMatchStatisticsRequestItem, IntApiDefect> v =
                    ItemValidationBuilder.of(item, IntApiDefect.class);
            v.item(item.getCampaignId(), GetRelevanceMatchStatisticsRequestItem.CAMPAIGN_ID)
                    .check(notNull())
                    .check(validId());
            v.item(item.getAdGroupId(), GetRelevanceMatchStatisticsRequestItem.AD_GROUP_ID)
                    .check(notNull())
                    .check(validId());
            v.item(item.getRelevanceMatchId(), GetRelevanceMatchStatisticsRequestItem.RELEVANCE_MATCH_ID)
                    .check(notNull())
                    .check(validId());
            return v.getResult();
        };
    }

    public ValidationResult<GetRetargetingStatisticsRequest, IntApiDefect> validate(
            GetRetargetingStatisticsRequest request) {
        ItemValidationBuilder<GetRetargetingStatisticsRequest, IntApiDefect> v =
                ItemValidationBuilder.of(request, IntApiDefect.class);

        v.check(notNull(), notEmptyRequest());
        if (!v.getResult().hasAnyErrors()) {
            v.item(request.getIntervalDays(), GetRetargetingStatisticsRequest.INTERVAL_DAYS)
                    .check(notNull())
                    .check(greaterThan(0));

            v.list(request.getSelectionCriteria(),
                    GetRetargetingStatisticsRequest.SELECTION_CRITERIA)
                    .check(notNull())
                    .check(notEmptyCollection())
                    .checkEach(notNull())
                    .checkEachBy(retargetingRequestItemValidator(), When.notNull());
        }
        return v.getResult();
    }

    private Validator<GetRetargetingStatisticsRequestItem, IntApiDefect> retargetingRequestItemValidator() {
        return item -> {
            ItemValidationBuilder<GetRetargetingStatisticsRequestItem, IntApiDefect> v =
                    ItemValidationBuilder.of(item, IntApiDefect.class);
            v.item(item.getCampaignId(), GetRetargetingStatisticsRequestItem.CAMPAIGN_ID)
                    .check(notNull())
                    .check(validId());
            v.item(item.getAdGroupId(), GetRetargetingStatisticsRequestItem.AD_GROUP_ID)
                    .check(notNull())
                    .check(validId());
            v.list(item.getRetargetingConditionIds(), GetRetargetingStatisticsRequestItem.RETARGETING_CONDITION_IDS)
                    .check(notNull())
                    .checkEach(notNull())
                    .checkEach(validId());
            return v.getResult();
        };
    }

    public ValidationResult<GetDynamicStatisticsRequest, IntApiDefect> validate(
            GetDynamicStatisticsRequest request) {
        ItemValidationBuilder<GetDynamicStatisticsRequest, IntApiDefect> v =
                ItemValidationBuilder.of(request, IntApiDefect.class);

        v.check(notNull(), notEmptyRequest());
        if (!v.getResult().hasAnyErrors()) {
            v.item(request.getIntervalDays(), GetDynamicStatisticsRequest.INTERVAL_DAYS)
                    .check(notNull())
                    .check(greaterThan(0));

            v.list(request.getSelectionCriteria(),
                    GetDynamicStatisticsRequest.SELECTION_CRITERIA)
                    .check(notNull())
                    .check(notEmptyCollection())
                    .checkEach(notNull())
                    .checkEachBy(dynamicRequestItemValidator(), When.notNull());
        }
        return v.getResult();
    }

    private Validator<GetDynamicStatisticsRequestItem, IntApiDefect> dynamicRequestItemValidator() {
        return item -> {
            ItemValidationBuilder<GetDynamicStatisticsRequestItem, IntApiDefect> v =
                    ItemValidationBuilder.of(item, IntApiDefect.class);
            v.item(item.getCampaignId(), GetDynamicStatisticsRequestItem.CAMPAIGN_ID)
                    .check(notNull())
                    .check(validId());
            v.item(item.getAdGroupId(), GetDynamicStatisticsRequestItem.AD_GROUP_ID)
                    .check(notNull())
                    .check(validId());
            v.list(item.getDynamicConditionIds(), GetDynamicStatisticsRequestItem.DYNAMIC_CONDITION_IDS)
                    .check(notNull())
                    .checkEach(notNull())
                    .checkEach(validId());
            return v.getResult();
        };
    }

    public ValidationResult<GetPerformanceStatisticsRequest, IntApiDefect> validate(
            GetPerformanceStatisticsRequest request) {
        ItemValidationBuilder<GetPerformanceStatisticsRequest, IntApiDefect> v =
                ItemValidationBuilder.of(request, IntApiDefect.class);

        v.check(notNull(), notEmptyRequest());
        if (!v.getResult().hasAnyErrors()) {
            v.item(request.getIntervalDays(), GetPerformanceStatisticsRequest.INTERVAL_DAYS)
                    .check(notNull())
                    .check(greaterThan(0));

            v.list(request.getSelectionCriteria(),
                    GetPerformanceStatisticsRequest.SELECTION_CRITERIA)
                    .check(notNull())
                    .check(notEmptyCollection())
                    .checkEach(notNull())
                    .checkEachBy(performanceRequestItemValidator(), When.notNull());
        }
        return v.getResult();
    }

    private Validator<GetPerformanceStatisticsRequestItem, IntApiDefect> performanceRequestItemValidator() {
        return item -> {
            ItemValidationBuilder<GetPerformanceStatisticsRequestItem, IntApiDefect> v =
                    ItemValidationBuilder.of(item, IntApiDefect.class);
            v.item(item.getCampaignId(), GetPerformanceStatisticsRequestItem.CAMPAIGN_ID)
                    .check(notNull())
                    .check(validId());
            v.item(item.getAdGroupId(), GetPerformanceStatisticsRequestItem.AD_GROUP_ID)
                    .check(notNull())
                    .check(validId());
            v.list(item.getPerformanceFilterIds(), GetPerformanceStatisticsRequestItem.PERFORMANCE_FILTER_IDS)
                    .check(notNull())
                    .checkEach(notNull())
                    .checkEach(validId());
            return v.getResult();
        };
    }

    public void validateAndCheck(GetOrdersStatByIntervalRequest request) {
        checkResult(validate(request));
    }

    public ValidationResult<GetOrdersStatByIntervalRequest, Defect> validate(GetOrdersStatByIntervalRequest request) {
        ItemValidationBuilder<GetOrdersStatByIntervalRequest, Defect> v
                = ItemValidationBuilder.of(request, Defect.class);

        v.list(request.getOrderIds(), GetOrdersStatByIntervalRequest.ORDER_IDS)
                .check(CommonConstraints.notNull())
                .checkEach(CommonConstraints.notNull())
                .checkEach(CommonConstraints.validId());

        v.item(request.getStartDate(), GetOrdersStatByIntervalRequest.START_DATE)
                .check(CommonConstraints.notNull())
                .check(isNotBeforeThan(BEGIN_OF_TIME_FOR_STAT), When.isValid());

        v.item(request.getEndDate(), GetOrdersStatByIntervalRequest.END_DATE)
                .check(CommonConstraints.notNull())
                .check(isNotBeforeThan(BEGIN_OF_TIME_FOR_STAT), When.isValid());

        Predicate<GetOrdersStatByIntervalRequest> endDateIsNotBeforeStart
                = r -> endDateIsNotBeforeStart(r.getStartDate(), r.getEndDate());
        v.check(Constraint.fromPredicate(endDateIsNotBeforeStart, CommonDefects.inconsistentState()),
                When.valueIs(r -> r.getEndDate() != null && r.getStartDate() != null));

        return v.getResult();
    }

    public void validateAndCheckIds(List<Long> ids, String fieldName) {
        ItemValidationBuilder<Collection<Long>, Defect> v = ItemValidationBuilder.of(ids, Defect.class);
        v.list(ids, fieldName)
                .check(CommonConstraints.notNull())
                .checkEach(CommonConstraints.notNull())
                .checkEach(CommonConstraints.validId());
        checkResult(v.getResult());
    }

    public void validateAndCheckPeriods(List<Period> periods, String fieldName) {
        ItemValidationBuilder<Collection<Period>, Defect> v = ItemValidationBuilder.of(periods, Defect.class);

        v.list(periods, fieldName)
                .check(CommonConstraints.notNull())
                .checkEachBy(this::validatePeriod);

        checkResult(v.getResult());
    }

    private ValidationResult<Period, Defect> validatePeriod(Period period) {
        ItemValidationBuilder<Period, Defect> v = ItemValidationBuilder.of(period, Defect.class);

        v.item(period.getKey(), Period.KEY_FIELD)
                .check(CommonConstraints.notNull());

        v.item(period.getStartDate(), Period.START_DATE_FIELD)
                .check(CommonConstraints.notNull())
                .check(isNotBeforeThan(BEGIN_OF_TIME_FOR_STAT), When.isValid());

        v.item(period.getEndDate(), Period.END_DATE_FIELD)
                .check(CommonConstraints.notNull())
                .check(isNotBeforeThan(BEGIN_OF_TIME_FOR_STAT), When.isValid());

        Predicate<Period> endDateIsNotBeforeStart
                = p -> endDateIsNotBeforeStart(p.getStartDate(), p.getEndDate());
        v.check(Constraint.fromPredicate(endDateIsNotBeforeStart, CommonDefects.inconsistentState()),
                When.valueIs(p -> p.getStartDate() != null && p.getEndDate() != null));

        return v.getResult();
    }

    private void checkResult(ValidationResult<?, Defect> validationResult) {
        IntapiValidationResponse intapiValidationResponse =
                validationResultConversionService.buildValidationResponse(validationResult);
        if (validationResult.hasAnyErrors()) {
            throw new IntApiException(HttpStatus.BAD_REQUEST, intapiValidationResponse);
        }
    }

    private Boolean endDateIsNotBeforeStart(LocalDate start, LocalDate end) {
        return !end.isBefore(start);
    }
}
