package ru.yandex.direct.intapi.entity.strategy;

import java.util.List;

import javax.annotation.Nonnull;
import javax.ws.rs.POST;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.direct.common.logging.EventType;
import ru.yandex.direct.common.logging.LoggingConfig;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.service.CampaignService;
import ru.yandex.direct.core.entity.campaign.service.CampaignStrategyService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.model.UidAndClientId;
import ru.yandex.direct.intapi.ErrorResponse;
import ru.yandex.direct.intapi.entity.strategy.model.SaveStrategyRequest;
import ru.yandex.direct.intapi.entity.strategy.model.SaveStrategyRequestConverter;
import ru.yandex.direct.intapi.entity.strategy.model.StrategyResponse;
import ru.yandex.direct.intapi.logging.ClientIdParam;
import ru.yandex.direct.intapi.logging.OperatorUid;
import ru.yandex.direct.intapi.validation.kernel.ValidationResultConversionService;
import ru.yandex.direct.intapi.validation.model.IntapiResponse;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.tvm.AllowServices;

import static java.util.Collections.singletonList;
import static ru.yandex.direct.tvm.TvmService.DIRECT_DEVELOPER;
import static ru.yandex.direct.tvm.TvmService.DIRECT_WEB_PROD;
import static ru.yandex.direct.tvm.TvmService.DIRECT_WEB_TEST;
import static ru.yandex.direct.utils.CommonUtils.nvl;

@RestController
@RequestMapping(value = "strategy", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
@Api(tags = "strategy")
@AllowServices(production = {DIRECT_WEB_PROD}, testing = {DIRECT_WEB_TEST, DIRECT_DEVELOPER})
public class StrategyController {
    private final ValidationResultConversionService validationResultConversionService;
    private final CampaignService campaignService;
    private final CampaignStrategyService campaignStrategyService;

    public StrategyController(ValidationResultConversionService validationResultConversionService,
                              CampaignService campaignService, CampaignStrategyService campaignStrategyService) {
        this.validationResultConversionService = validationResultConversionService;
        this.campaignService = campaignService;
        this.campaignStrategyService = campaignStrategyService;
    }

    @POST
    @ApiOperation(
            value = "сохранение стратегий на Java",
            nickname = "saveStrategy",
            httpMethod = "POST"
    )
    @ApiResponses({
            @ApiResponse(code = 403, message = "Недостаточно прав для выполнения операции",
                    response = ErrorResponse.class),
            @ApiResponse(code = 400, message = "Некорректный запрос",
                    response = ErrorResponse.class)
    })
    @RequestMapping(path = "/save_strategy",
            method = RequestMethod.POST,
            consumes = javax.ws.rs.core.MediaType.APPLICATION_JSON,
            produces = javax.ws.rs.core.MediaType.APPLICATION_JSON
    )
    @LoggingConfig(logRequestBody = EventType.ALL, logResponseBody = EventType.ALL)
    public IntapiResponse saveStrategy(
            @RequestParam(value = "operator_uid") @OperatorUid long operatorUid,
            @RequestParam(value = "client_id") @ClientIdParam long clientId,
            @RequestBody @Nonnull SaveStrategyRequest body) {
        List<Campaign> campaigns = campaignService.getCampaigns(ClientId.fromLong(clientId),
                singletonList(body.getCid()));
        if (campaigns.isEmpty()) {
            return StrategyResponse.failed("campaign with cid= " + body.getCid() + " not found");
        }
        Campaign campaign = campaigns.get(0);
        MassResult<Long> result = campaignStrategyService.updateTextCampaignStrategy(body.getCid(),
                SaveStrategyRequestConverter.toDbStrategy(body),
                operatorUid, UidAndClientId.of(campaign.getUserId(), ClientId.fromLong(campaign.getClientId())),
                nvl(body.getAttributionModelChanged(), false));
        return validationResultConversionService.buildValidationResponse(result);
    }
}
