package ru.yandex.direct.intapi.entity.tracelogs.controller;

import java.time.LocalDate;
import java.time.LocalDateTime;

import javax.ws.rs.core.MediaType;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiParam;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.direct.intapi.entity.tracelogs.model.profilestats.ProfileStatsRequest;
import ru.yandex.direct.intapi.entity.tracelogs.model.profilestats.ProfileStatsResponse;
import ru.yandex.direct.intapi.entity.tracelogs.model.profilestatsbyela.ProfileStatsByElaRequest;
import ru.yandex.direct.intapi.entity.tracelogs.model.profilestatsbyela.ProfileStatsByElaResponse;
import ru.yandex.direct.intapi.entity.tracelogs.service.profilestats.ProfileStatsService;
import ru.yandex.direct.intapi.entity.tracelogs.service.profilestatsbyela.ProfileStatsByElaService;

/**
 * Ручка для работы с "сырыми" Trace логами
 */
@RestController
@RequestMapping("trace_logs")
@Api(tags = "intapiTraceLogs", value = "работа с \"сырыми\" Trace логами")
public class TraceLogsController {
    private final ProfileStatsByElaService profileStatsByElaService;
    private final ProfileStatsService profileStatsService;

    @Autowired
    public TraceLogsController(
            ProfileStatsService profileStatsService, ProfileStatsByElaService profileStatsByElaService
    ) {
        this.profileStatsService = profileStatsService;
        this.profileStatsByElaService = profileStatsByElaService;
    }

    /**
     * Ручка принимает и отдает JSON
     *
     * @param request {@link ProfileStatsRequest}
     * @return {@link ProfileStatsResponse}
     */
    @PostMapping(
            path = "profile_stats",
            consumes = MediaType.APPLICATION_JSON,
            produces = MediaType.APPLICATION_JSON
    )
    @ResponseBody
    public ProfileStatsResponse getProfileStats(@ApiParam(required = true) @RequestBody ProfileStatsRequest request) {
        LocalDateTime timeTo = request.getTimeTo();
        LocalDateTime timeFrom = request.getTimeFrom();
        Long timePeriod = request.getTimePeriod();

        if (timePeriod != null) {
            timeTo = profileStatsService.getDateTimeTZ(request.getTimeZone());
            timeFrom = timeTo.minusSeconds(timePeriod);
        } else {
            if (timeFrom == null) {
                throw new IllegalArgumentException("At least time_period or time_from must be specified");
            }
            if (timeTo == null) {
                timeTo = profileStatsService.getDateTimeTZ(request.getTimeZone());
            }
        }

        if (request.getGroupBy().isEmpty()) {
            throw new IllegalArgumentException("You must group by at least one field");
        }

        if (timeFrom.isAfter(timeTo)) {
            throw new IllegalArgumentException("timeFrom is after timeTo");
        }

        return profileStatsService.getProfileLogs(request, timeFrom, timeTo);
    }

    /**
     * Ручка принимает и отадет JSON
     *
     * @param request {@link ProfileStatsByElaRequest}
     * @return {@link ProfileStatsByElaResponse}
     */
    @PostMapping(
            path = "profile_stats_by_ela",
            consumes = MediaType.APPLICATION_JSON,
            produces = MediaType.APPLICATION_JSON
    )
    @ResponseBody
    public ProfileStatsByElaResponse getProfileStatsByEla(
            @ApiParam(required = true) @RequestBody ProfileStatsByElaRequest request
    ) {
        LocalDate lastDate = profileStatsByElaService.getDateTZ(request.getTimezone());

        LocalDate dateFrom = request.getDateFrom();
        LocalDate dateTo = request.getDateTo();
        if (dateFrom == null) {
            if (dateTo != null) {
                dateFrom = dateTo;
            } else {
                dateFrom = lastDate;
            }
        }

        if (dateTo == null) {
            dateTo = dateFrom;
        }

        return profileStatsByElaService.getProfileStatsByEla(request, dateFrom, dateTo);
    }
}
