package ru.yandex.direct.intapi.entity.tracelogs.model.profilestatsbyela;

import java.time.LocalDate;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Класс для аггрегации финального результата по profileStatsByEla
 */
public class AggRows {
    private List<Map<LocalDate, AggRow>> data;
    private List<String> regexpNames;
    private List<String> regexps;

    public AggRows(List<String> regexpNames, List<String> regexps) {
        this.regexpNames = regexpNames;
        this.regexps = regexps;

        this.data = new ArrayList<>();
        for (int i = 0; i < regexps.size(); i++) {
            data.add(new HashMap<>());
        }
    }

    /**
     * Аггрегация FuncStats по дате
     */
    public void aggregateFuncStats(List<FuncStats> stats) {
        for (FuncStats funcStats : stats) {
            for (int i = 0; i < data.size(); i++) {
                if (funcStats.getCmdMatch().get(i) > 0) {
                    Map<LocalDate, AggRow> map = data.get(i);
                    if (!map.containsKey(funcStats.getDate())) {
                        map.put(funcStats.getDate(), new AggRow());
                    }

                    map.get(funcStats.getDate()).addFuncStats(funcStats);
                }
            }
        }
    }

    /**
     * Аггрегация Cmds и Summary по дате
     */
    public void aggregateCmdsSummary(List<CmdsSummary> cmdsSummaries) {
        Map<CmdsSummary, Cmd> mergedCmdSummary = new HashMap<>();
        List<Summary> summaries = new ArrayList<>();
        for (CmdsSummary cmdsSummary : cmdsSummaries) {
            if (!mergedCmdSummary.containsKey(cmdsSummary)) {
                mergedCmdSummary.put(cmdsSummary, cmdsSummary.getCmd());
            } else {
                mergedCmdSummary.get(cmdsSummary).addCnt(cmdsSummary.getCmd().getCnt());
            }

            summaries.add(cmdsSummary.getSummary());
        }

        aggregateCmds(mergedCmdSummary.values());
        aggregateSummaries(summaries);
    }

    private void aggregateCmds(Collection<Cmd> cmds) {
        for (Cmd cmd : cmds) {
            for (int i = 0; i < data.size(); i++) {
                if (cmd.getCmdMatch().get(i) > 0) {
                    Map<LocalDate, AggRow> map = data.get(i);
                    if (!map.containsKey(cmd.getDate())) {
                        map.put(cmd.getDate(), new AggRow());
                    }

                    map.get(cmd.getDate()).addCmd(cmd);
                }
            }
        }
    }

    private void aggregateSummaries(List<Summary> summaries) {
        for (Summary summary : summaries) {
            for (int i = 0; i < data.size(); i++) {
                if (summary.getCmdMatch().get(i) > 0) {
                    Map<LocalDate, AggRow> map = data.get(i);
                    if (!map.containsKey(summary.getDate())) {
                        map.put(summary.getDate(), new AggRow());
                    }

                    map.get(summary.getDate()).addSummary(summary);
                }
            }
        }
    }

    /**
     * Слияние Cmds, FuncStats и Summary по дате и regexp'у в финальный результат
     */
    public List<ProfileStatsByElaResponse.Stats> mergeResults(List<LocalDate> dates) {
        List<ProfileStatsByElaResponse.Stats> results = new ArrayList<>();
        for (int i = 0; i < regexpNames.size(); i++) {
            for (LocalDate date : dates) {
                AggRow row = data.get(i).get(date);

                if (row == null) {
                    results.add(new ProfileStatsByElaResponse.Stats(regexps.get(i), date));
                } else {
                    List<Cmd> cmds = row.getCmds();
                    cmds.sort(Collections.reverseOrder(Comparator.comparingInt(Cmd::getCnt)));

                    Map<Float, Summary> mergedSummary = mergeSummary(row);

                    List<Summary> sortedMergedSummary = new ArrayList<>(mergedSummary.values());
                    sortedMergedSummary.sort(Comparator.comparingDouble(Summary::getRoundedEla));

                    Map<String, Map<String, Float>> mergedFuncsEla = mergeFuncsEla(row);

                    ProfileStatsByElaResponse.Stats result = new ProfileStatsByElaResponse.Stats(
                            regexps.get(i), date, cmds, sortedMergedSummary, mergedFuncsEla);
                    results.add(result);
                }
            }
        }

        return results;
    }

    private Map<Float, Summary> mergeSummary(AggRow row) {
        Map<Float, Summary> mergedSummary = new HashMap<>();
        for (Summary rowSummary : row.getSummary()) {
            if (rowSummary.getRoundedEla() > 0) {
                if (mergedSummary.containsKey(rowSummary.getRoundedEla())) {
                    mergedSummary.get(rowSummary.getRoundedEla()).addCnt(rowSummary.getCnt());
                } else {
                    mergedSummary.put(rowSummary.getRoundedEla(), rowSummary);
                }
            }
        }

        return mergedSummary;
    }

    private Map<String, Map<String, Float>> mergeFuncsEla(AggRow row) {
        Map<String, Map<String, Float>> mergedFuncsEla = new HashMap<>();
        for (FuncStats funcStats : row.getFuncStats()) {
            Map<String, Float> elaToProc;
            if (mergedFuncsEla.containsKey(funcStats.getFuncParam())) {
                elaToProc = mergedFuncsEla.get(funcStats.getFuncParam());
            } else {
                elaToProc = new HashMap<>();
            }
            elaToProc.put(
                    Float.toString(funcStats.getRoundedEla()),
                    elaToProc.getOrDefault(Float.toString(funcStats.getRoundedEla()), 0.0f) + funcStats.getElaProc()
            );
            mergedFuncsEla.put(funcStats.getFuncParam(), elaToProc);
        }

        return mergedFuncsEla;
    }

    private class AggRow {
        private List<Cmd> cmds;
        private List<Summary> summary;
        private List<FuncStats> funcStats;

        private AggRow() {
            cmds = new ArrayList<>();
            summary = new ArrayList<>();
            funcStats = new ArrayList<>();
        }

        private void addCmd(Cmd cmd) {
            this.cmds.add(cmd);
        }

        private void addSummary(Summary summary) {
            this.summary.add(summary);
        }

        private void addFuncStats(FuncStats funcStats) {
            this.funcStats.add(funcStats);
        }

        private List<Cmd> getCmds() {
            return cmds;
        }

        private List<Summary> getSummary() {
            return summary;
        }

        private List<FuncStats> getFuncStats() {
            return funcStats;
        }
    }
}
