package ru.yandex.direct.intapi.entity.tracelogs.service.profilestats;

import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.clickhouse.SqlBuilder;
import ru.yandex.direct.dbutil.wrapper.DatabaseWrapperProvider;
import ru.yandex.direct.intapi.entity.tracelogs.model.profilestats.FilterField;
import ru.yandex.direct.intapi.entity.tracelogs.model.profilestats.GroupByField;
import ru.yandex.direct.intapi.entity.tracelogs.model.profilestats.ProfileData;
import ru.yandex.direct.intapi.entity.tracelogs.model.profilestats.ProfileStatsRequest;
import ru.yandex.direct.intapi.entity.tracelogs.model.profilestats.ProfileStatsResponse;

import static ru.yandex.direct.dbutil.wrapper.SimpleDb.CLICKHOUSE_CLOUD;
import static ru.yandex.direct.intapi.entity.tracelogs.model.profilestats.ProfileData.FORMATTER;

/**
 * Класс-сервис для {@link ru.yandex.direct.intapi.entity.tracelogs.controller.TraceLogsController}
 */
@Service
public class ProfileStatsService {
    static final String FUNC_ELA = "func_ela";
    static final String FUNC_CNT = "func_cnt";
    static final String FUNC_OBJ_NUM = "func_obj_num";
    static final String FUNC_OBJ_NUM_1 = "func_obj_num_1";
    static final String ELA = "ela";
    static final String CNT = "cnt";
    static final String CPU_USER = "cpu_user";
    static final String CPU_SYSTEM = "cpu_system";
    static final String MEM = "mem";

    private static final String FUNC_ELA_AVG = "func_ela_avg";
    private static final String FUNC_OBJ_ELA_AVG = "func_obj_ela_avg";
    private static final String ELA_AVG = "ela_avg";

    private final DatabaseWrapperProvider dbProvider;

    @Autowired
    public ProfileStatsService(DatabaseWrapperProvider dbProvider) {
        this.dbProvider = dbProvider;
    }

    /**
     * Получить profile статистику
     */
    public ProfileStatsResponse getProfileLogs(
            ProfileStatsRequest request, LocalDateTime timeFrom, LocalDateTime timeTo
    ) {
        SqlBuilder builder = new SqlBuilder().from("trace");

        builder.where(
                "log_date BETWEEN toDate(?) AND toDate(?)",
                DateTimeFormatter.ISO_LOCAL_DATE.format(timeFrom.toLocalDate()),
                DateTimeFormatter.ISO_LOCAL_DATE.format(timeTo.toLocalDate())
        );

        builder.where("log_time >= toDateTime(?)", FORMATTER.format(timeFrom));
        builder.where("log_time < toDateTime(?)", FORMATTER.format(timeTo));

        List<ProfileStatsResponse.ResultField> resultFields = new ArrayList<>();

        ProfileStatsRowMapper mapRower = new ProfileStatsRowMapper();

        fillGroupBy(request, builder, resultFields, mapRower);

        applyFilters(request, builder, mapRower);

        addCalculatedFields(builder, resultFields, mapRower.isUseFuncs());

        List<ProfileData> data = dbProvider.get(CLICKHOUSE_CLOUD).query(
                builder.toString(), builder.getBindings(), mapRower);

        return new ProfileStatsResponse(null, resultFields, data);
    }

    public LocalDateTime getDateTimeTZ(ZoneId timezone) {
        return LocalDateTime.now(timezone);
    }

    private void fillGroupBy(ProfileStatsRequest request, SqlBuilder builder,
                             List<ProfileStatsResponse.ResultField> resultFields, ProfileStatsRowMapper mapRower) {
        String sql;
        for (GroupByField field : request.getGroupBy()) {
            if (field == GroupByField.STAT_TIME) {
                long timeAgg = request.getTimeAgg();
                sql = String.format("toDateTime(intDiv(toUInt32(log_time), %d) * %d)", timeAgg, timeAgg);
            } else {
                sql = field.getSqlValue();
            }

            builder.selectExpression(sql, field.toString().toLowerCase());

            builder.groupBy(field.toString().toLowerCase());

            if (field == GroupByField.FUNC || field == GroupByField.FUNC_PARAM) {
                mapRower.setUseFuncs(true);
            }

            resultFields.add(new ProfileStatsResponse.ResultField(
                    field.name().toLowerCase(), field.name().toLowerCase(), true));
        }
    }

    private void applyFilters(ProfileStatsRequest request, SqlBuilder builder, ProfileStatsRowMapper mapRower) {
        String sql;
        for (Map.Entry<FilterField, List<String>> entry : request.getFilters().entrySet()) {
            sql = entry.getKey().getSqlValue();
            if (entry.getValue() == null) {
                continue;
            }

            List<String> likePatterns = new ArrayList<>();
            List<String> likeQueryParts = new ArrayList<>();
            for (String pattern : entry.getValue()) {
                likeQueryParts.add(String.format("%s LIKE ?", sql));
                likePatterns.add(pattern);
            }

            builder.where(String.format("(%s)", String.join(" OR ", likeQueryParts)), likePatterns.toArray());

            if (entry.getKey() == FilterField.FUNC || entry.getKey() == FilterField.FUNC_PARAM) {
                mapRower.setUseFuncs(true);
            }
        }
    }

    private void addCalculatedFields(
            SqlBuilder builder,
            List<ProfileStatsResponse.ResultField> resultFields,
            boolean useFuncs
    ) {
        if (useFuncs) {
            builder
                    .selectExpression("sum(profile.all_ela)", FUNC_ELA)
                    .selectExpression("sum(profile.calls)", FUNC_CNT)
                    .selectExpression("sum(profile.obj_num)", FUNC_OBJ_NUM)
                    .selectExpression("sum(if(profile.obj_num > 0, profile.obj_num, 1))", FUNC_OBJ_NUM_1)
                    .orderBy(FUNC_ELA, SqlBuilder.Order.DESC)
                    .arrayJoin("profile");

            resultFields.add(new ProfileStatsResponse.ResultField(FUNC_ELA, FUNC_ELA));
            resultFields.add(new ProfileStatsResponse.ResultField(FUNC_CNT, FUNC_CNT));
            resultFields.add(new ProfileStatsResponse.ResultField(FUNC_OBJ_NUM, FUNC_OBJ_NUM));
            resultFields.add(new ProfileStatsResponse.ResultField(FUNC_ELA_AVG, FUNC_ELA_AVG));
            resultFields.add(new ProfileStatsResponse.ResultField(FUNC_OBJ_ELA_AVG, FUNC_OBJ_ELA_AVG));
        } else {
            builder
                    .selectExpression("sum(ela)", ELA)
                    .selectExpression("count(*)", CNT)
                    .selectExpression("sum(cpu_user)", CPU_USER)
                    .selectExpression("sum(cpu_system)", CPU_SYSTEM)
                    .selectExpression("sum(mem)", MEM)
                    .orderBy(ELA, SqlBuilder.Order.DESC);

            resultFields.add(new ProfileStatsResponse.ResultField(ELA, ELA));
            resultFields.add(new ProfileStatsResponse.ResultField(CNT, CNT));
            resultFields.add(new ProfileStatsResponse.ResultField(ELA_AVG, ELA_AVG));
            resultFields.add(new ProfileStatsResponse.ResultField(CPU_USER, CPU_USER));
            resultFields.add(new ProfileStatsResponse.ResultField(CPU_SYSTEM, CPU_SYSTEM));
            resultFields.add(new ProfileStatsResponse.ResultField(MEM, MEM));
        }
    }
}
