package ru.yandex.direct.intapi.entity.tracelogs.service.profilestatsbyela;

import java.time.LocalDate;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import com.google.common.base.Preconditions;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.clickhouse.ClickHouseUtil;
import ru.yandex.direct.clickhouse.SqlBuilder;
import ru.yandex.direct.dbutil.wrapper.DatabaseWrapperProvider;
import ru.yandex.direct.intapi.entity.tracelogs.model.profilestatsbyela.AggRows;
import ru.yandex.direct.intapi.entity.tracelogs.model.profilestatsbyela.CmdsSummary;
import ru.yandex.direct.intapi.entity.tracelogs.model.profilestatsbyela.FuncStats;
import ru.yandex.direct.intapi.entity.tracelogs.model.profilestatsbyela.ProfileStatsByElaRequest;
import ru.yandex.direct.intapi.entity.tracelogs.model.profilestatsbyela.ProfileStatsByElaResponse;

import static ru.yandex.direct.dbutil.wrapper.SimpleDb.CLICKHOUSE_CLOUD;

/**
 * Класс-сервис для {@link ru.yandex.direct.intapi.entity.tracelogs.controller.TraceLogsController}
 */
@Service
public class ProfileStatsByElaService {
    static final String LOG_DATE = "log_date";
    static final String CMD_TYPE = "cmd_type";
    static final String CMD = "cmd";
    static final String CNT = "cnt";
    static final String ELA_SUM = "ela_sum";
    static final String CPU_USER = "cpu_user";
    static final String CPU_SYSTEM = "cpu_system";
    static final String MEM = "mem";
    static final String ROUNDED_ELA = "rounded_ela";
    static final String FUNC_PARAM = "func_param";
    static final String ELA_PROC = "ela_proc";

    private static final String SERVICE = "service";
    private static final String METHOD = "method";
    private static final String TRACE = "trace";
    private static final String DATE_BETWEEN = "log_date BETWEEN toDate(?) AND toDate(?)";
    private static final String DATE_IN = "log_date IN (?, ?)";

    private static final float MAX_ROUNDED_ELA = 60000f;

    private final DatabaseWrapperProvider dbProvider;

    @Autowired
    public ProfileStatsByElaService(DatabaseWrapperProvider dbProvider) {
        this.dbProvider = dbProvider;
    }

    public LocalDate getDateTZ(ZoneId timezone) {
        return LocalDate.now(timezone);
    }

    /**
     * Получить статистику profile by ela
     */
    public ProfileStatsByElaResponse getProfileStatsByEla(
            ProfileStatsByElaRequest request, LocalDate dateFrom, LocalDate dateTo
    ) {
        List<LocalDate> dates = getDateRange(request.isCompareDates(), dateTo, dateFrom);

        List<String> regexps = request.getRegexps();
        if (regexps == null || regexps.isEmpty()) {
            return new ProfileStatsByElaResponse(null, Collections.emptyList());
        }

        List<String> regexpNames = IntStream
                .range(0, regexps.size())
                .boxed()
                .map(i -> String.format("cmdmatch%d", i))
                .collect(Collectors.toList());
        String regexpWhere = String.format("(%s)", String.join(" OR ", regexpNames));

        AggRows aggRows = new AggRows(regexpNames, regexps);

        String roundedElaQuery = makeRoundElaStmt(request.getBoundaries());

        List<CmdsSummary> allCmdsSummary = getCmdsSummary(regexps, regexpNames, dateFrom, dateTo, regexpWhere,
                roundedElaQuery, request.isCompareDates());
        aggRows.aggregateCmdsSummary(allCmdsSummary);

        List<FuncStats> allFuncs = getAllFuncs(regexps, regexpNames, dateFrom, dateTo, regexpWhere, roundedElaQuery,
                request.isCompareDates());
        aggRows.aggregateFuncStats(allFuncs);

        return new ProfileStatsByElaResponse(null, aggRows.mergeResults(dates));
    }

    private List<LocalDate> getDateRange(boolean isCompareDates, LocalDate dateTo, LocalDate dateFrom) {
        if (isCompareDates) {
            if (!dateFrom.equals(dateTo)) {
                return Arrays.asList(dateFrom, dateTo);
            } else {
                return Collections.singletonList(dateFrom);
            }
        } else {
            return enumerateDateRange(dateFrom, dateTo);
        }
    }

    private List<LocalDate> enumerateDateRange(LocalDate dateFrom, LocalDate dateTo) {
        Preconditions.checkArgument(dateFrom.equals(dateTo) || dateFrom.isBefore(dateTo), "timeFrom is after timeTo");

        List<LocalDate> dates = new ArrayList<>();
        while (dateFrom.compareTo(dateTo) <= 0) {
            dates.add(dateFrom);
            dateFrom = dateFrom.plusDays(1);
        }

        return dates;
    }

    private List<CmdsSummary> getCmdsSummary(List<String> regexps, List<String> regexpNames, LocalDate dateFrom,
                                             LocalDate dateTo,
                                             String regexpWhere, String roundedElaQuery, boolean isCompareDates) {
        SqlBuilder builder = new SqlBuilder();
        selectCmd(regexps, regexpNames, builder);
        builder
                .select(LOG_DATE)
                .selectExpression(SERVICE, CMD_TYPE)
                .selectExpression(METHOD, CMD)
                .selectExpression(roundedElaQuery, ROUNDED_ELA)
                .selectExpression("count(profile.calls)", CNT)
                .selectExpression("sum(ela)", ELA_SUM)
                .selectExpression("sum(cpu_user)", CPU_USER)
                .selectExpression("sum(cpu_system)", CPU_SYSTEM)
                .selectExpression("sum(mem)", MEM)
                .from(TRACE)
                .where(regexpWhere)
                .groupBy(LOG_DATE, CMD_TYPE, CMD, ROUNDED_ELA);

        filterDates(builder, dateFrom, dateTo, isCompareDates);

        CmdSummaryRowMapper cmdSummaryRowMapper = new CmdSummaryRowMapper(regexpNames);

        return dbProvider.get(CLICKHOUSE_CLOUD).query(builder.toString(), builder.getBindings(), cmdSummaryRowMapper);
    }

    private List<FuncStats> getAllFuncs(List<String> regexps, List<String> regexpNames,
                                        LocalDate dateFrom, LocalDate dateTo, String regexpWhere,
                                        String roundedElaQuery, boolean isCompareDates) {
        SqlBuilder builder = new SqlBuilder();
        selectCmd(regexps, regexpNames, builder);
        builder
                .select(LOG_DATE)
                .selectExpression("concat(concat(profile.func, '#'), profile.tags)", FUNC_PARAM)
                .selectExpression(roundedElaQuery, ROUNDED_ELA)
                .selectExpression("sum(profile.all_ela - profile.childs_ela)", ELA_PROC)
                .from(TRACE)
                .arrayJoin("profile")
                .where(regexpWhere)
                .groupBy(LOG_DATE, SERVICE, METHOD, FUNC_PARAM, ROUNDED_ELA);

        filterDates(builder, dateFrom, dateTo, isCompareDates);

        FuncRowMapper funcRowMapper = new FuncRowMapper(regexpNames);

        return dbProvider.get(CLICKHOUSE_CLOUD).query(builder.toString(), builder.getBindings(), funcRowMapper);
    }

    private void filterDates(SqlBuilder builder, LocalDate dateFrom, LocalDate dateTo, boolean isCompareDates) {
        if (isCompareDates) {
            builder.where(
                    DATE_IN,
                    DateTimeFormatter.ISO_LOCAL_DATE.format(dateFrom),
                    DateTimeFormatter.ISO_LOCAL_DATE.format(dateTo)
            );
        } else {
            builder.where(
                    DATE_BETWEEN,
                    DateTimeFormatter.ISO_LOCAL_DATE.format(dateFrom),
                    DateTimeFormatter.ISO_LOCAL_DATE.format(dateTo)
            );
        }
    }

    private String makeRoundElaStmt(List<Float> boundaries) {
        if (boundaries.isEmpty()) {
            return Float.toString(MAX_ROUNDED_ELA);
        }

        StringBuilder roundElaStmt = new StringBuilder("multiIf(");
        for (Float boundary : boundaries) {
            roundElaStmt.append(String.format(Locale.US, "ela < %f, %f, ", boundary, boundary));
        }
        roundElaStmt.append(MAX_ROUNDED_ELA).append(")");

        return roundElaStmt.toString();
    }

    private void selectCmd(List<String> regexps, List<String> regexpNames, SqlBuilder builder) {
        for (int i = 0; i < regexps.size(); i++) {
            builder.selectExpression(
                    String.format("like(concat(concat(service, '/'), method), '%s')",
                            ClickHouseUtil.escape(regexps.get(i))),
                    regexpNames.get(i)
            );
        }
    }
}
