package ru.yandex.direct.intapi.entity.turboecom;

import java.util.List;
import java.util.Locale;
import java.util.Set;

import javax.ws.rs.POST;
import javax.ws.rs.core.MediaType;

import com.google.common.collect.ImmutableSet;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.direct.bannerstorage.client.BannerStorageClient;
import ru.yandex.direct.bannerstorage.client.BannerStorageClientException;
import ru.yandex.direct.bannerstorage.client.model.Creative;
import ru.yandex.direct.bannerstorage.client.model.Item;
import ru.yandex.direct.bannerstorage.client.model.ItemsRequest;
import ru.yandex.direct.bannerstorage.client.model.Parameter;
import ru.yandex.direct.common.logging.EventType;
import ru.yandex.direct.common.logging.LoggingConfig;
import ru.yandex.direct.intapi.ErrorResponse;
import ru.yandex.direct.intapi.IntApiException;
import ru.yandex.direct.intapi.client.IntApiClient;
import ru.yandex.direct.intapi.client.IntApiClientException;
import ru.yandex.direct.intapi.entity.turboecom.model.TransferMoneyRequest;
import ru.yandex.direct.intapi.entity.turboecom.model.TransferMoneyResponse;
import ru.yandex.direct.intapi.entity.turboecom.service.TurboEcomService;
import ru.yandex.direct.intapi.entity.turboecom.service.TurboEcomValidationService;
import ru.yandex.direct.intapi.logging.OperatorUid;
import ru.yandex.direct.intapi.validation.kernel.ValidationResultConversionService;
import ru.yandex.direct.tvm.AllowServices;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.tvm.TvmService.DIRECT_DEVELOPER;
import static ru.yandex.direct.tvm.TvmService.DIRECT_INTAPI_TEST;
import static ru.yandex.direct.tvm.TvmService.TURBO_ECOM_PROD;
import static ru.yandex.direct.tvm.TvmService.TURBO_ECOM_TEST;

@RestController
@RequestMapping("retail_recommendations")
@Api(tags = "retail_recommendations", value = "для работы турбо рекомендаций DIRECT-111215")
@AllowServices(
        production = {TURBO_ECOM_PROD, DIRECT_DEVELOPER},
        testing = {TURBO_ECOM_TEST, DIRECT_DEVELOPER, DIRECT_INTAPI_TEST})
public class TurboEcomController {

    private final IntApiClient intApiClient;
    private final BannerStorageClient bannerStorageClient;
    private final TurboEcomService turboEcomService;
    private final TurboEcomValidationService turboEcomValidationService;
    private final ValidationResultConversionService validationResultConversionService;

    // TODO(dimitrovsd): перенести константу отсюда в TurboEcomValidationService
    //создавать драфтовые креативы без модерации можно только для этого uidа
    public static final Set<Long> ACCEPTED_UIDS = ImmutableSet.of(1018113692L); //yndx-smart-recommend-agency

    @Autowired
    public TurboEcomController(IntApiClient intApiClient,
                               BannerStorageClient bannerStorageClient,
                               TurboEcomService turboEcomService,
                               TurboEcomValidationService turboEcomValidationService,
                               ValidationResultConversionService validationResultConversionService) {
        this.intApiClient = intApiClient;
        this.bannerStorageClient = bannerStorageClient;
        this.turboEcomService = turboEcomService;
        this.turboEcomValidationService = turboEcomValidationService;
        this.validationResultConversionService = validationResultConversionService;
    }

    @POST
    @ApiOperation(
            value = "Создание стаб-креатива для смарт кампании",
            nickname = "createStubCreative",
            httpMethod = "POST"
    )
    @ApiResponses({
            @ApiResponse(code = 403, message = "Недостаточно прав для выполнения операции",
                    response = ErrorResponse.class),
            @ApiResponse(code = 400, message = "Некорректный запрос",
                    response = ErrorResponse.class)
    })
    @RequestMapping(path = "create_stub_creative",
            method = RequestMethod.POST,
            consumes = MediaType.APPLICATION_JSON,
            produces = MediaType.APPLICATION_JSON
    )
    @LoggingConfig(logRequestBody = EventType.ALL, logResponseBody = EventType.ALL)
    public Creative createStubCreative(
            @RequestParam(value = "operator_uid") @OperatorUid Long operatorUid,
            @RequestParam(value = "client_login") String clientLogin) {
        if (!ACCEPTED_UIDS.contains(operatorUid)) {
            throw new IntApiException(HttpStatus.BAD_REQUEST, "Данный operator_uid не доступен");
        }

        //минимальный необходимый набор для создания и модерации креатива
        //1014 шаблон из тикета https://st.yandex-team.ru/MEDIATEMPLATES-224
        //остальные значения не важны, т.к. креатив показываться не будет
        Creative creative = new Creative()
                .withName("Креатив для товарных рекомендаций")
                .withTemplateId(1014)
                .withTag("turbo_ecom")
                .withParameters(List.of(new Parameter()
                        .withParamName("LINK")
                        .withValues(List.of("https://turbo.site"))));
        ItemsRequest tnsArticles = new ItemsRequest().withItems(List.of(new Item().withId(-3)));
        ItemsRequest tnsBrands = new ItemsRequest().withItems(List.of(new Item().withId(1)));
        int creativeId;
        try {
            creativeId = bannerStorageClient.createCreative(creative, Locale.ENGLISH).getId();
            bannerStorageClient.addTnsArticles(creativeId, tnsArticles);
            bannerStorageClient.addTnsBrands(creativeId, tnsBrands);
            bannerStorageClient.requestModeration(creativeId);
        } catch (BannerStorageClientException e) {
            throw new IntApiException(HttpStatus.UNPROCESSABLE_ENTITY, "Ошибка создания креатива: %s", e.getMessage(),
                    e);
        }
        try {

            intApiClient.changeNotifyCreatives(clientLogin, operatorUid, List.of(creativeId));
        } catch (IntApiClientException e) {
            throw new IntApiException(HttpStatus.UNPROCESSABLE_ENTITY, "Ошибка синхронизации креатива: %s",
                    e.getMessage(), e);
        }

        return creative.withId(creativeId);
    }

    @POST
    @RequestMapping(path = "transfer_money",
            method = RequestMethod.POST,
            consumes = MediaType.APPLICATION_JSON,
            produces = MediaType.APPLICATION_JSON
    )
    public TransferMoneyResponse transferMoney(
            @RequestBody TransferMoneyRequest transferMoneyRequest) {
        ValidationResult<TransferMoneyRequest, Defect> vr = turboEcomValidationService.validate(transferMoneyRequest);
        if (vr.hasAnyErrors()) {
            throw new IntApiException(HttpStatus.BAD_REQUEST, new ErrorResponse(ErrorResponse.ErrorCode.BAD_PARAM,
                    validationResultConversionService.convertErrorsToString(vr)));
        }

        return turboEcomService.transferMoney(transferMoneyRequest);
    }
}
