package ru.yandex.direct.intapi.entity.turboecom.service;

import java.math.BigDecimal;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import ru.yandex.direct.balance.client.model.request.createtransfermultiple.CreateTransferMultipleRequest;
import ru.yandex.direct.balance.client.model.request.createtransfermultiple.TransferTargetFrom;
import ru.yandex.direct.balance.client.model.request.createtransfermultiple.TransferTargetTo;
import ru.yandex.direct.core.entity.campaign.model.Wallet;
import ru.yandex.direct.core.entity.campaign.model.WalletCampaign;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.repository.WalletRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.intapi.entity.turboecom.model.TransferMoneyRequest;

import static com.google.common.base.Preconditions.checkState;
import static java.util.function.Function.identity;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * WARNING: Компонент не для общего использования. Содержит логику только для турбо-рекомендаций.
 */
@Component
class TransferMoneyRequestConverter {

    /**
     * Количество денег в рублях, которое будем переводить на каждый кошелек
     */
    private static final double TRANSFER_SUM = 0.01;

    private final ShardHelper shardHelper;
    private final WalletRepository walletRepository;
    private final CampaignRepository campaignRepository;

    private final int directServiceId;

    @Autowired
    TransferMoneyRequestConverter(@Value("${balance.directServiceId}") int directServiceId,
                                  ShardHelper shardHelper,
                                  WalletRepository walletRepository,
                                  CampaignRepository campaignRepository) {
        this.directServiceId = directServiceId;

        this.shardHelper = shardHelper;
        this.walletRepository = walletRepository;
        this.campaignRepository = campaignRepository;
    }

    public CreateTransferMultipleRequest convert(TransferMoneyRequest transferMoneyRequest) {
        ClientId clientIdFrom = ClientId.fromLong(transferMoneyRequest.getClientIdFrom());
        List<ClientId> clientIdsTo = mapList(transferMoneyRequest.getClientIdsTo(), ClientId::fromLong);

        return new CreateTransferMultipleRequestBuilder()
                .withOperatorUid(transferMoneyRequest.getOperatorUid())
                .withClientIdFrom(clientIdFrom)
                .withClientIdsTo(clientIdsTo)
                .build();
    }

    private class CreateTransferMultipleRequestBuilder {

        private Long operatorUid;
        private ClientId clientIdFrom;
        private Collection<ClientId> clientIdsTo;

        private Collection<Long> walletIdsTo;
        private WalletCampaign walletCampaignFrom;

        private CreateTransferMultipleRequestBuilder withOperatorUid(Long operatorUid) {
            this.operatorUid = operatorUid;
            return this;
        }

        private CreateTransferMultipleRequestBuilder withClientIdFrom(ClientId clientIdFrom) {
            this.clientIdFrom = clientIdFrom;
            return this;
        }

        private CreateTransferMultipleRequestBuilder withClientIdsTo(Collection<ClientId> clientIdsTo) {
            this.clientIdsTo = clientIdsTo;
            return this;
        }

        private CreateTransferMultipleRequest build() {
            checkState(operatorUid != null && clientIdFrom != null && clientIdsTo != null);
            fillAdditionalFields();
            checkState(walletCampaignFrom != null && walletIdsTo != null);

            BigDecimal walletFromSum = walletCampaignFrom.getSum();
            BigDecimal walletFromSumAfter =
                    walletFromSum.subtract(BigDecimal.valueOf(TRANSFER_SUM * walletIdsTo.size()));

            return new CreateTransferMultipleRequest()
                    .withOperatorUid(operatorUid)
                    .withTransferTargetsFrom(List.of(
                            new TransferTargetFrom()
                                    .withServiceId(directServiceId)
                                    .withCampaignId(walletCampaignFrom.getId())
                                    .withQtyOld(walletFromSum)
                                    .withQtyNew(walletFromSumAfter)))
                    .withTransferTargetsTo(mapList(walletIdsTo, walletIdTo ->
                            new TransferTargetTo()
                                    .withCampaignId(walletIdTo)
                                    .withServiceId(directServiceId)
                                    .withQtyDelta(BigDecimal.valueOf(TRANSFER_SUM))));
        }

        private void fillAdditionalFields() {
            Map<ClientId, Wallet> walletByClientId = getWalletByClientId();

            this.walletCampaignFrom = getWalletCampaignFrom(walletByClientId);
            this.walletIdsTo = getWalletIdsTo(walletByClientId);
        }

        private Map<ClientId, Wallet> getWalletByClientId() {
            List<ClientId> allClientIds = StreamEx.of(clientIdsTo)
                    .append(clientIdFrom)
                    .toList();

            return shardHelper.groupByShard(allClientIds, ShardKey.CLIENT_ID)
                    .stream()
                    .mapKeyValue(walletRepository::getAllWalletExistingCampByClientId)
                    .flatMap(StreamEx::of)
                    .mapToEntry(Wallet::getClientId, identity())
                    .toMap();
        }

        private WalletCampaign getWalletCampaignFrom(Map<ClientId, Wallet> walletByClientId) {
            int shard = shardHelper.getShardByClientId(clientIdFrom);
            Wallet walletFrom = walletByClientId.get(clientIdFrom);
            return campaignRepository.getWalletsByWalletCampaignIds(shard, List.of(walletFrom.getWalletCampaignId()))
                    .iterator().next();
        }

        private List<Long> getWalletIdsTo(Map<ClientId, Wallet> walletByClientId) {
            return StreamEx.of(clientIdsTo)
                    .map(walletByClientId::get)
                    .map(Wallet::getWalletCampaignId)
                    .toList();
        }
    }
}
