package ru.yandex.direct.intapi.entity.turboecom.service;

import java.util.Set;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.intapi.entity.turboecom.model.TransferMoneyRequest;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.Collections.emptySet;
import static ru.yandex.direct.intapi.entity.turboecom.TurboEcomController.ACCEPTED_UIDS;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.notEmptyCollection;
import static ru.yandex.direct.validation.constraint.CommonConstraints.inSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notInSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;

@Service
public class TurboEcomValidationService {

    private final RbacService rbacService;
    private final ShardHelper shardHelper;

    @Autowired
    public TurboEcomValidationService(RbacService rbacService,
                                      ShardHelper shardHelper) {
        this.rbacService = rbacService;
        this.shardHelper = shardHelper;
    }

    public ValidationResult<TransferMoneyRequest, Defect> validate(TransferMoneyRequest request) {
        ItemValidationBuilder<TransferMoneyRequest, Defect> ivb = ItemValidationBuilder.of(request);

        Set<Long> safeClientIdsTo = nvl(listToSet(request.getClientIdsTo()), emptySet());
        Set<Long> accessibleAgencySubclientIds = getAccessibleAgencySubclientIds(request.getOperatorUid());

        // TODO(dimitrovsd): вынести легкие проверки в превалидацию
        ivb.item(request.getOperatorUid(), "operatorUid")
                .check(notNull())
                .check(inSet(ACCEPTED_UIDS));
        ivb.item(request.getClientIdFrom(), "clientIdFrom")
                .check(notNull())
                .check(validId())
                .check(notInSet(safeClientIdsTo))
                .check(inSet(accessibleAgencySubclientIds), When.isValid());
        ivb.list(request.getClientIdsTo(), "clientIdTo")
                .check(notNull())
                .checkEach(notNull())
                .check(notEmptyCollection())
                .checkEach(validId())
                .checkEach(inSet(accessibleAgencySubclientIds), When.isValid());

        return ivb.getResult();
    }

    private Set<Long> getAccessibleAgencySubclientIds(Long operatorUid) {
        if (operatorUid == null) {
            return emptySet();
        }

        Set<Long> accessibleAgencySubclientUids = listToSet(rbacService.getAgencySubclients(operatorUid));
        return listToSet(shardHelper.getClientIdsByUids(accessibleAgencySubclientUids).values());
    }
}
