package ru.yandex.direct.intapi.entity.turbolandings.service;

import java.util.List;
import java.util.Map;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.turbolanding.model.TurboLanding;
import ru.yandex.direct.core.entity.turbolanding.model.TurboLandingWithCountersAndGoals;
import ru.yandex.direct.core.entity.turbolanding.service.TurboLandingService;
import ru.yandex.direct.core.entity.turbolanding.service.UpdateCounterGrantsService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.intapi.ErrorResponse;
import ru.yandex.direct.intapi.IntApiException;
import ru.yandex.direct.intapi.entity.turbolandings.model.IntapiTurboLanding;
import ru.yandex.direct.intapi.entity.turbolandings.model.InterfaceTypeEnum;
import ru.yandex.direct.intapi.entity.turbolandings.model.SaveTurbolandingsResult;
import ru.yandex.direct.intapi.entity.turbolandings.model.SaveTurbolandingsSuccessResponse;
import ru.yandex.direct.intapi.entity.turbolandings.model.TurboLandingInfoResponse;
import ru.yandex.direct.intapi.entity.turbolandings.model.UpdateCounterGrantsIntapiRequestItem;
import ru.yandex.direct.intapi.entity.turbolandings.validation.SaveTurboLandingsValidationService;
import ru.yandex.direct.intapi.entity.turbolandings.validation.UpdateCounterGrantsValidationService;
import ru.yandex.direct.intapi.validation.IntApiDefect;
import ru.yandex.direct.intapi.validation.kernel.ValidationResultConversionService;
import ru.yandex.direct.intapi.validation.model.IntapiResponse;
import ru.yandex.direct.intapi.validation.model.IntapiSuccessResponse;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.intapi.entity.turbolandings.service.TurboLandingConverter.toTurboLandingWithCountersAndGoals;
import static ru.yandex.direct.intapi.validation.ValidationUtils.checkResult;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
public class TurboLandingsService {

    private final TurboLandingService turboLandingService;
    private final UpdateCounterGrantsService updateCounterGrantsService;
    private final UpdateCounterGrantsValidationService updateCounterGrantsValidationService;
    private final RbacService rbacService;
    private final SaveTurboLandingsValidationService saveTurboLandingsValidationService;
    private final FeatureService featureService;
    private final ValidationResultConversionService validationResultConversionService;

    @Autowired
    public TurboLandingsService(TurboLandingService turboLandingService,
                                UpdateCounterGrantsService updateCounterGrantsService,
                                UpdateCounterGrantsValidationService updateCounterGrantsValidationService,
                                RbacService rbacService,
                                SaveTurboLandingsValidationService saveTurboLandingsValidationService,
                                FeatureService featureService,
                                ValidationResultConversionService validationResultConversionService
    ) {
        this.turboLandingService = turboLandingService;
        this.updateCounterGrantsService = updateCounterGrantsService;
        this.updateCounterGrantsValidationService = updateCounterGrantsValidationService;
        this.rbacService = rbacService;
        this.saveTurboLandingsValidationService = saveTurboLandingsValidationService;
        this.featureService = featureService;
        this.validationResultConversionService = validationResultConversionService;
    }

    public IntapiResponse updateCounterGrants(Long operatorUid, List<UpdateCounterGrantsIntapiRequestItem> params) {
        ValidationResult<List<UpdateCounterGrantsIntapiRequestItem>, IntApiDefect> validationResult =
                updateCounterGrantsValidationService.validate(params);
        checkResult(validationResult);

        Map<Long, List<Long>> userIdsByCounterId = StreamEx.of(params)
                .toMap(UpdateCounterGrantsIntapiRequestItem::getCounterId,
                        UpdateCounterGrantsIntapiRequestItem::getUserIds);

        updateCounterGrantsService.addCountersToResyncQueue(operatorUid, userIdsByCounterId);
        return new IntapiSuccessResponse();
    }

    public IntapiResponse saveTurboLanding(long operatorUid, ClientId clientId, List<IntapiTurboLanding> request) {
        checkPermission(operatorUid, clientId);

        ValidationResult<List<IntapiTurboLanding>, IntApiDefect> validationResult =
                saveTurboLandingsValidationService.validate(request);
        checkResult(validationResult);

        List<TurboLandingWithCountersAndGoals> turboLandings = mapList(request,
                intapiTurbolanding -> toTurboLandingWithCountersAndGoals(clientId, intapiTurbolanding));

        if (turboLandings.stream().anyMatch(TurboLanding::getIsCpaModerationRequired)) {
            return saveTurbolandingsWithValidation(operatorUid, clientId, turboLandings);
        }
        turboLandingService.saveTurboLandings(operatorUid, clientId, turboLandings);
        return new IntapiSuccessResponse();
    }

    public IntapiResponse saveTurbolandingsWithValidation(long operatorUid, ClientId clientId,
                                                          List<TurboLandingWithCountersAndGoals> turboLandings) {
        MassResult<Long> result = turboLandingService.saveModeratableTurbolanding(operatorUid, clientId, turboLandings);

        return new SaveTurbolandingsSuccessResponse(EntryStream
                .zip(mapList(turboLandings, TurboLanding::getId), result.getResult())
                .mapKeyValue(this::convertToSaveTurbolandingResult)
                .toList());
    }

    public SaveTurbolandingsResult convertToSaveTurbolandingResult(Long turbolandingId, Result<Long> result) {
        if (result.isSuccessful()) {
            return SaveTurbolandingsResult.ok(turbolandingId);
        } else {
            return SaveTurbolandingsResult.error(turbolandingId,
                    validationResultConversionService.convertErrorsToString(result.getValidationResult()));
        }
    }

    public TurboLandingInfoResponse getTurboLandingInfo(long operatorUid, ClientId clientId, Long turbolandingId) {
        checkPermission(operatorUid, clientId);

        InterfaceTypeEnum interfaceType = featureService.isEnabledForClientId(clientId, FeatureName.GRID) ?
                InterfaceTypeEnum.GRID : InterfaceTypeEnum.CLASSIC;

        int attachedBannersCount =
                turboLandingService.getAttachedBannerIds(clientId, turbolandingId).size();

        return new TurboLandingInfoResponse()
                .withBannersCount(attachedBannersCount)
                .withInterfaceType(interfaceType);
    }

    private void checkPermission(long operatorUid, ClientId clientId) {
        long clientChiefUid = rbacService.getChiefByClientId(clientId);
        if (!rbacService.isOwner(operatorUid, clientChiefUid)) {
            throw new IntApiException(HttpStatus.FORBIDDEN,
                    new ErrorResponse(ErrorResponse.ErrorCode.PERMISSION_DENIED, "Permission denied"));
        }
    }
}
