package ru.yandex.direct.intapi.entity.turbolandings.validation;

import java.util.List;

import org.springframework.stereotype.Service;

import ru.yandex.direct.intapi.entity.turbolandings.model.IntapiTurboLanding;
import ru.yandex.direct.intapi.entity.turbolandings.model.IntapiTurboLandingCounter;
import ru.yandex.direct.intapi.entity.turbolandings.model.IntapiTurboLandingGoal;
import ru.yandex.direct.intapi.validation.IntApiDefect;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.constraint.UniqueItemsConstraint;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.intapi.validation.IntApiConstraints.notEmptyCollection;
import static ru.yandex.direct.intapi.validation.IntApiConstraints.notNull;
import static ru.yandex.direct.intapi.validation.IntApiConstraints.validId;


@Service
public class SaveTurboLandingsValidationService {

    public ValidationResult<List<IntapiTurboLanding>, IntApiDefect> validate(List<IntapiTurboLanding> turboLandings) {
        return ListValidationBuilder.of(turboLandings, IntApiDefect.class)
                .check(notNull())
                .check(notEmptyCollection())
                .checkEach(notNull())
                .checkEachBy(this::validateTurboLanding, When.notNull())
                .checkEach(new UniqueItemsConstraint<>(IntapiTurboLanding::getId,
                        () -> IntApiDefect.notParameterized("duplicate id")), When.isValid())
                .getResult();
    }

    private ValidationResult<IntapiTurboLanding, IntApiDefect> validateTurboLanding(IntapiTurboLanding turboLanding) {
        ItemValidationBuilder<IntapiTurboLanding, IntApiDefect> vb =
                ItemValidationBuilder.of(turboLanding, IntApiDefect.class);

        vb.item(turboLanding.getId(), IntapiTurboLanding.ID)
                .check(notNull())
                .check(validId());

        vb.item(turboLanding.getId(), IntapiTurboLanding.VERSION)
                .check(notNull());

        vb.item(turboLanding.getName(), IntapiTurboLanding.NAME)
                .check(notNull());

        vb.item(turboLanding.getUrl(), IntapiTurboLanding.URL)
                .check(notNull());

        vb.list(turboLanding.getCounters(), IntapiTurboLanding.COUNTERS)
                .check(notNull())
                .checkEach(notNull())
                .checkEachBy(this::validateTurboLandingCounter, When.notNull())
                .checkEach(new UniqueItemsConstraint<>(IntapiTurboLandingCounter::getId,
                        () -> IntApiDefect.notParameterized("duplicate counterId")), When.isValid());

        return vb.getResult();
    }

    private ValidationResult<IntapiTurboLandingCounter, IntApiDefect> validateTurboLandingCounter(
            IntapiTurboLandingCounter counter) {
        ItemValidationBuilder<IntapiTurboLandingCounter, IntApiDefect> vb =
                ItemValidationBuilder.of(counter, IntApiDefect.class);

        vb.item(counter.getId(), IntapiTurboLandingCounter.ID)
                .check(notNull());

        vb.list(counter.getGoals(), IntapiTurboLandingCounter.GOALS)
                .check(notNull())
                .checkEachBy(this::validateTurboLandingGoal);

        vb.item(counter.getIsUserCounter(), IntapiTurboLandingCounter.IS_USER_COUNTER)
                .check(notNull());

        return vb.getResult();
    }

    private ValidationResult<IntapiTurboLandingGoal, IntApiDefect> validateTurboLandingGoal(
            IntapiTurboLandingGoal goal) {
        ItemValidationBuilder<IntapiTurboLandingGoal, IntApiDefect> vb =
                ItemValidationBuilder.of(goal, IntApiDefect.class);

        vb.item(goal.getId(), IntapiTurboLandingGoal.ID)
                .check(notNull());

        return vb.getResult();
    }
}
